package eu.dnetlib.data.hadoop;

import java.io.IOException;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.client.Scan;
import org.apache.hadoop.hbase.util.Bytes;
import org.springframework.beans.factory.annotation.Autowired;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

import eu.dnetlib.data.hadoop.config.ClusterName;
import eu.dnetlib.data.hadoop.config.ConfigurationEnumerator;
import eu.dnetlib.data.hadoop.rmi.HadoopServiceException;
import eu.dnetlib.data.hadoop.utils.HadoopUtils;

public class HadoopServiceCore {

	private static final Log log = LogFactory.getLog(HadoopServiceCore.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Autowired
	private HadoopClientMap clients;

	@Autowired
	protected ConfigurationEnumerator configurationEnumerator;

	public List<String> listTables(final ClusterName clusterName) throws IOException, HadoopServiceException {
		HBaseAdmin admin = getHBaseAdmin(clusterName);
		return Lists.newArrayList(Iterables.transform(Lists.newArrayList(admin.listTables()), HadoopUtils.tableName()));
	}

	private HBaseAdmin getHBaseAdmin(final ClusterName clusterName) throws HadoopServiceException {
		HBaseAdmin admin = clients.getHbaseAdmin(clusterName);

		if (admin == null) throw new HadoopServiceException(String.format("HBase admin not available for cluster: '%s'", clusterName.toString()));

		return admin;
	}

	public void truncateTable(final ClusterName clusterName, final String table) throws IOException, HadoopServiceException {
		HBaseAdmin admin = getHBaseAdmin(clusterName);

		if (!admin.tableExists(table)) throw new IllegalStateException("cannot truncate unexisting table");

		final HTableDescriptor desc = admin.getTableDescriptor(table.getBytes());

		log.info("disabling table: " + table);
		admin.disableTable(table);

		log.info("deleting table: " + table);
		admin.deleteTable(table);

		log.info("creating table: " + table);
		admin.createTable(desc);
	}

	public boolean existTable(final ClusterName clusterName, final String table) throws IOException, HadoopServiceException {
		HBaseAdmin admin = getHBaseAdmin(clusterName);

		return admin.tableExists(table);
	}

	public void dropTable(final ClusterName clusterName, final String table) throws IOException, HadoopServiceException {
		HBaseAdmin admin = getHBaseAdmin(clusterName);

		if (!admin.tableExists(table)) throw new IllegalStateException("cannot drop unexisting table: '" + table + "'");

		log.info("disabling table: " + table);
		admin.disableTable(table);

		log.info("deleting table: " + table);
		admin.deleteTable(table);
	}

	public void createTable(final ClusterName clusterName, final String table, final Set<String> columns) throws IOException, HadoopServiceException {
		HBaseAdmin admin = getHBaseAdmin(clusterName);

		if (admin.tableExists(table)) throw new IllegalStateException("table already exists");

		HTableDescriptor desc = new HTableDescriptor(table);
		for (String column : columns) {
			HColumnDescriptor hds = new HColumnDescriptor(column);
			hds.setMaxVersions(1);
			desc.addFamily(hds);
		}
		log.info("creating hbase table: " + table);

		admin.createTable(desc);
		log.info("created hbase table: [" + table + "] descriptor: [" + desc.toString() + "]");
	}

	public void ensureTable(final ClusterName clusterName, final String table, final Set<String> columns) throws IOException, HadoopServiceException {

		HBaseAdmin admin = getHBaseAdmin(clusterName);

		if (!admin.tableExists(table)) {
			createTable(clusterName, table, columns);
		} else {
			final HTableDescriptor desc = admin.getTableDescriptor(Bytes.toBytes(table));
			final Set<String> foundColumns = Sets.newHashSet(Iterables.transform(Lists.newArrayList(desc.getColumnFamilies()), HadoopUtils.columnName()));

			log.info("ensuring columns on table " + table + ": " + columns);
			final List<String> missingColumns = Lists.newArrayList(Iterables.filter(columns, HadoopUtils.columnPredicate(foundColumns)));
			if (!missingColumns.isEmpty()) {

				if (admin.isTableEnabled(table)) {
					admin.disableTable(table);
				}

				for (String column : missingColumns) {
					log.info("hbase table: '" + table + "', adding column: " + column);
					admin.addColumn(table, new HColumnDescriptor(column));
				}

				admin.enableTable(table);
			}
		}
	}

	public void write(final ClusterName clusterName, final String tableName, final List<Put> puts) throws IOException {
		final Configuration conf = configurationEnumerator.get(clusterName);
		final HTable table = new HTable(conf, tableName);
		try {
			table.put(puts);
		} finally {
			table.flushCommits();
			table.close();
		}
	}

	public void delete(final ClusterName clusterName, final String tableName, final List<Delete> deletes) throws IOException {
		final Configuration conf = configurationEnumerator.get(clusterName);
		final HTable table = new HTable(conf, tableName);
		try {
			table.delete(deletes);
		} finally {
			table.flushCommits();
			table.close();
		}
	}

	public Result getRow(final ClusterName clusterName, final String tableName, final byte[] id) throws IOException {
		final Configuration conf = configurationEnumerator.get(clusterName);
		final HTable table = new HTable(conf, tableName);
		try {
			return table.get(new Get(id));
		} finally {
			table.close();
		}
	}

	public List<Result> getRows(final ClusterName clusterName, final String tableName, final Scan scan) throws IOException {
		final Configuration conf = configurationEnumerator.get(clusterName);
		final HTable table = new HTable(conf, tableName);
		try {
			final ResultScanner rs = table.getScanner(scan);
			try {
				return Lists.newArrayList(rs.iterator());
			} finally {
				rs.close();
			}
		} finally {
			table.close();
		}
	}

	public Configuration getClusterConiguration(final ClusterName clusterName) {
		return configurationEnumerator.get(clusterName);
	}

}
