package eu.dnetlib.data.hadoop.action;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.mapreduce.CopyTable;
import org.apache.hadoop.mapred.JobConf;
import org.apache.hadoop.mapred.RunningJob;

import com.google.common.base.Joiner;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import eu.dnetlib.data.hadoop.HadoopJob;
import eu.dnetlib.data.hadoop.config.ClusterName;
import eu.dnetlib.data.hadoop.mapred.MapreduceJobMonitor;
import eu.dnetlib.data.hadoop.rmi.HadoopServiceException;
import eu.dnetlib.data.hadoop.utils.JobProfile;
import eu.dnetlib.data.hadoop.utils.ScanProperties;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJob;
import eu.dnetlib.enabling.tools.blackboard.BlackboardServerHandler;

/**
 * The Class SubmitAdminJobAction.
 */
public class SubmitAdminJobAction extends SubmitMapreduceJobAction {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(SubmitAdminJobAction.class); // NOPMD by marko on 11/24/08 5:02 PM

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * eu.dnetlib.data.hadoop.action.SubmitMapreduceJobAction#executeAsync(eu.dnetlib.enabling.tools.blackboard.BlackboardServerHandler,
	 * eu.dnetlib.enabling.tools.blackboard.BlackboardJob)
	 */
	@Override
	public void executeAsync(final BlackboardServerHandler handler, final BlackboardJob bbJob) {
		final String jobName = bbJob.getParameters().get("job.name");
		final ClusterName clusterName = ClusterName.valueOf(bbJob.getParameters().get("cluster"));

		try {
			JobProfile jobProfile = loadISJobConfiguration(jobName, bbJob.getParameters());

			validateJobParams(handler, bbJob, jobName, jobProfile);

			JobConf jobConf = prepareJob(getConf(clusterName), jobName, jobProfile, bbJob.getParameters());

			if (!clientMap.isMapreduceAvailable(clusterName))
				throw new HadoopServiceException("mapreduce not available for cluster: " + clusterName.toString());

			logJobDetails(jobConf);

			final RunningJob runningJob = clientMap.getJtClient(clusterName).submitJob(jobConf);
			final String jobId = newJobId(clusterName, runningJob.getID().getId());

			jobRegistry.registerJob(HadoopJob.newInstance(jobId, clusterName, jobProfile,
					new MapreduceJobMonitor(runningJob, newCompletionCallback(handler, bbJob, jobId))));

			updateJobStatus(jobName);
			handler.ongoing(bbJob);
		} catch (Throwable e) {
			log.error("error executing hadoop job: " + jobName, e);
			handler.failed(bbJob, e);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see eu.dnetlib.data.hadoop.action.SubmitMapreduceJobAction#prepareJob(org.apache.hadoop.conf.Configuration, java.lang.String,
	 * eu.dnetlib.data.hadoop.utils.JobProfile, java.util.Map)
	 */
	@Override
	protected JobConf prepareJob(final Configuration configuration, final String jobName, final JobProfile jobProfile, final Map<String, String> parameters)
			throws IOException, HadoopServiceException {

		log.info("creating job: " + jobName);

		JobConf jobConf = new JobConf(configuration);
		jobConf.setJobName(jobName);
		jobConf.set("dnet.mapred.job.description", jobProfile.getDescription());

		String jobLib = getJobLib(configuration, jobProfile);
		jobConf.setJar(new Path(jobLib).toString());

		set(jobConf, jobProfile.getJobDefinition());
		set(jobConf, parameters);

		if (jobName.equals("copytable")) return copyTable(jobProfile, parameters, jobConf);

		throw new HadoopServiceException("unknown admin job: " + jobName);
	}

	/**
	 * Builds a CopyTable job. Mimics
	 *
	 * bin/hbase org.apache.hadoop.hbase.mapreduce.CopyTable --starttime=1265875194289 --endtime=1265878794289
	 * --peer.adr=server1,server2,server3:2181:/hbase --families=myOldCf:myNewCf,cf2,cf3 TestTable
	 *
	 * @param jobProfile
	 *            the job profile
	 * @param parameters
	 *            the parameters
	 * @param jobConf
	 *            the job conf
	 * @return the job conf
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	private JobConf copyTable(final JobProfile jobProfile, final Map<String, String> parameters, final JobConf jobConf) throws IOException {

		ScanProperties scan = jobProfile.getScanProperties();

		List<String> params = Lists.newArrayList();
		if (!scan.getFamilies().isEmpty()) {
			params.add("--families=" + Joiner.on(",").join(scan.getFamilies()));
		}
		if (parameters.containsKey("hbase.mapred.output.quorum")) {
			params.add("--peer.addr=" + parameters.get("hbase.mapred.output.quorum"));
		}
		params.add(parameters.get("hbase.mapreduce.outputtable"));

		Configuration copyConf = CopyTable.createSubmittableJob(jobConf, Iterables.toArray(params, String.class)).getConfiguration();

		return merge(jobConf, copyConf);
	}

	/**
	 * Merge.
	 *
	 * @param jobConf
	 *            the job conf
	 * @param copyConf
	 *            the copy conf
	 * @return the job conf
	 */
	private JobConf merge(final JobConf jobConf, final Configuration copyConf) {
		for (Entry<String, String> e : copyConf) {
			jobConf.set(e.getKey(), e.getValue());
		}
		return jobConf;
	}

}
