package eu.dnetlib.data.hadoop.action;

import java.io.IOException;
import java.util.List;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.util.Bytes;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

import eu.dnetlib.data.hadoop.config.ClusterName;
import eu.dnetlib.data.hadoop.config.ConfigurationEnumerator;
import eu.dnetlib.data.transform.Column;
import eu.dnetlib.data.transform.Row;
import eu.dnetlib.data.transform.XsltRowTransformerFactory;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;

public class HbaseTableFeeder {
	
	private int batchSize = 100;
	
	private ResultSetClientFactory resultSetClientFactory;
	
	@Autowired
	protected ConfigurationEnumerator configurationEnumerator;	
	
	public void feed(final String epr, final String xsl, final ClusterName clusterName, final String tableName, final ImportEprActionCallback callback) {
		
		final Thread feederThread = new Thread(new Runnable() {

			@Override
			public void run() {
				try {
					callback.done(doWrite(asRows(epr, xsl), getConf(clusterName), tableName));
				} catch (Throwable e) {
					callback.failed(e);
				}
			}
		});
		feederThread.start();
	}

	private int doWrite(final Iterable<Row> rows, final Configuration configuration, final String tableName) throws IOException {
		final List<Put> buffer = Lists.newArrayList();
		final HTable htable = new HTable(configuration, tableName);
		int count = 0;
		try {
			int i = 0;
			for (Row row : rows) {
				final Put put = new Put(Bytes.toBytes(row.getKey()));
	
				for (Column<String, byte[]> col : row) {
					put.add(Bytes.toBytes(row.getColumnFamily()), Bytes.toBytes(col.getName()), col.getValue());
				}
	
				buffer.add(put);
				
				if ((++i % getBatchSize()) == 0) {
					htable.put(buffer);
					count+=buffer.size();
					buffer.clear();
				}
			}
		} finally {
			if (!buffer.isEmpty()) {
				htable.put(buffer);
				count+=buffer.size();
			}
			htable.flushCommits();
			htable.close();
		}
		return count;
	}
	
	protected Iterable<Row> asRows(final String epr, final String xsl) {
		return Iterables.concat(Iterables.transform(resultSetClientFactory.getClient(epr), XsltRowTransformerFactory.newInstance(xsl)));
	}	
	
	protected Configuration getConf(ClusterName clusterName) {
		return configurationEnumerator.get(clusterName);
	}	

	public int getBatchSize() {
		return batchSize;
	}

	public void setBatchSize(int batchSize) {
		this.batchSize = batchSize;
	}
	
	public ResultSetClientFactory getResultSetClientFactory() {
		return resultSetClientFactory;
	}

	@Required
	public void setResultSetClientFactory(final ResultSetClientFactory resultSetClientFactory) {
		this.resultSetClientFactory = resultSetClientFactory;
	}	
	
}
