package eu.dnetlib.data.hadoop.oozie;

import java.util.Date;
import java.util.HashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.oozie.client.OozieClient;
import org.apache.oozie.client.OozieClientException;
import org.apache.oozie.client.WorkflowJob.Status;

import eu.dnetlib.data.hadoop.action.JobCompletion;
import eu.dnetlib.data.hadoop.action.JobMonitor;
import eu.dnetlib.data.hadoop.rmi.HadoopServiceException;

public class OozieJobMonitor extends JobMonitor {

	private static final Log log = LogFactory.getLog(JobMonitor.class); // NOPMD by marko on 11/24/08 5:02 PM

	private final OozieClient oozieClient;

	private final String jobId;

	public OozieJobMonitor(OozieClient oozieClient, String jobId, JobCompletion callback) {
		super(callback);
		this.oozieClient = oozieClient;
		this.jobId = jobId;
	}

	@Override
	public void run() {
		try {
			log.info("waiting for oozie job completion: " + getHadoopId());

			Status status = doGetStatus();
			while (status.equals(Status.RUNNING)) {
				Thread.sleep(monitorSleepTimeSecs * 1000);

				Status currentStatus = doGetStatus();
				if (!status.equals(currentStatus)) {
					status = currentStatus;
					lastActivity = new Date(System.currentTimeMillis());
				}
			}

			log.debug("job " + jobId + " finihsed with status: " + status);
			if (status == Status.SUCCEEDED) {
				// TODO set some content to return to the blackboard msg.
				getCallback().done(new HashMap<String, String>());
            } else {
				// TODO retrieve some failure information from the oozie client.
				String msg = "hadoop job: " + getHadoopId() + " failed with status: " + getStatus() + ", oozie log:\n "
						+ getOozieClient().getJobLog(getHadoopId()) + "\n\n";
				getCallback().failed(msg, new HadoopServiceException(msg));
            }
		} catch (Throwable e) {
			getCallback().failed(getHadoopId(), e);
		}
	}

	@Override
	public String getHadoopId() {
		return jobId;
	}

	public OozieClient getOozieClient() {
		return oozieClient;
	}

	@Override
	public String getStatus() {
		try {
			return doGetStatus().toString();
		} catch (OozieClientException e) {
			log.error("error accessing job status", e);
			return "UNKNOWN";
		}
	}

	private Status doGetStatus() throws OozieClientException {
		return getOozieClient().getJobInfo(getHadoopId()).getStatus();
	}

	@Override
	public Date getLastActivity() {
		return lastActivity;
	}

	@Override
	public Date getStartTime() throws HadoopServiceException {
		try {
			return getOozieClient().getJobInfo(getHadoopId()).getStartTime();
		} catch (OozieClientException e) {
			throw new HadoopServiceException("unable to read job start time", e);
		}
	}

	@Override
	public String getTrackerUrl() {
		return getOozieClient().getOozieUrl();
	}

	@Override
	public void kill() {
		try {
			getOozieClient().kill(getHadoopId());
		} catch (OozieClientException e) {
			log.error("unable to kill job: " + getHadoopId(), e);
		}
	}

}
