package eu.dnetlib.data.hadoop;

import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.mapred.JobClient;
import org.apache.oozie.client.OozieClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.context.annotation.Lazy;

import com.google.common.collect.Maps;
import com.google.gson.Gson;

import eu.dnetlib.data.hadoop.config.ClusterName;
import eu.dnetlib.data.hadoop.hbase.HBaseAdminFactory;
import eu.dnetlib.data.hadoop.oozie.OozieClientFactory;
import eu.dnetlib.data.hadoop.rmi.HadoopServiceException;
import eu.dnetlib.data.mapreduce.JobClientFactory;

public class HadoopClientMap {

	private static final Log log = LogFactory.getLog(HadoopClientMap.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Lazy
	@Autowired
	private JobClientFactory jobClientFactory;

	@Lazy
	@Autowired
	private OozieClientFactory oozieClientFactory;

	@Lazy
	@Autowired
	private HBaseAdminFactory hbaseAdminFactory;

	private Map<String, Map<String, String>> enabledClients = Maps.newHashMap();

	private final Map<ClusterName, HadoopClients> clients = Maps.newHashMap();

	private final Executor executor = Executors.newSingleThreadExecutor();

	public void init() {
		log.info("clients conf: " + getEnabledClients());
		for (final String name : enabledClients.keySet()) {
			executor.execute(new Runnable() {

				@Override
				public void run() {
					try {
						log.info("initializing clients for hadoop cluster: " + name);
						ClusterName clusterName = ClusterName.valueOf(name);

						Map<String, String> clientsConf = enabledClients.get(name);

						JobClient jobClient = Boolean.valueOf(clientsConf.get("mapred")) ? jobClientFactory.newInstance(name) : null;
						OozieClient oozieClient = Boolean.valueOf(clientsConf.get("oozie")) ? oozieClientFactory.newInstance(clusterName) : null;
						HBaseAdmin hbaseAdmin = Boolean.valueOf(clientsConf.get("hbase")) ? hbaseAdminFactory.newInstance(clusterName) : null;

						clients.put(clusterName, new HadoopClients(jobClient, oozieClient, hbaseAdmin));
					} catch (Exception e) {
						log.error("Error initializing hadoop client for cluster: " + name, e);
					}
				}
			});
		}
	}

	public JobClient getJtClient(final ClusterName name) {
		return getClients(name).getJtClient();
	}

	public boolean isMapreduceAvailable(final ClusterName name) {
		return getClients(name).isMapredAvailable();
	}

	public OozieClient getOozieClient(final ClusterName name) {
		return getClients(name).getOozieClient();
	}

	public boolean isOozieAvailable(final ClusterName name) {
		return getClients(name).isOozieAvailable();
	}

	public HBaseAdmin getHbaseAdmin(final ClusterName name) {
		return getClients(name).getHbaseAdmin();
	}

	public HadoopClients getClients(final ClusterName name) {
		final HadoopClients hadoopClients = clients.get(name);
		if (hadoopClients == null) throw new IllegalArgumentException("cluster " + name.toString() + " is currently disabled");
		return hadoopClients;
	}

	// //////////

	public String getEnabledClients() {
		return new Gson().toJson(enabledClients);
	}

	@Required
	@SuppressWarnings("unchecked")
	public void setEnabledClients(final String enabledClients) {
		this.enabledClients = new Gson().fromJson(enabledClients, Map.class);
	}

}
