package eu.dnetlib.data.hadoop.action;

import java.io.IOException;
import java.io.StringWriter;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.mapred.JobConf;
import org.apache.hadoop.mapred.RunningJob;

import eu.dnetlib.data.hadoop.HadoopJob;
import eu.dnetlib.data.hadoop.config.ClusterName;
import eu.dnetlib.data.hadoop.mapred.MapreduceJobMonitor;
import eu.dnetlib.data.hadoop.rmi.HadoopServiceException;
import eu.dnetlib.data.hadoop.utils.JobProfile;
import eu.dnetlib.data.hadoop.utils.ScanFactory;
import eu.dnetlib.data.hadoop.utils.ScanProperties;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJob;
import eu.dnetlib.enabling.tools.blackboard.BlackboardServerHandler;
import eu.dnetlib.miscutils.functional.xml.IndentXmlString;

public class SubmitMapreduceJobAction extends AbstractSubmitAction {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(SubmitMapreduceJobAction.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Override
	public void executeAsync(final BlackboardServerHandler handler, final BlackboardJob bbJob) {

		final String jobName = bbJob.getParameters().get("job.name");
		final ClusterName clusterName = ClusterName.valueOf(bbJob.getParameters().get("cluster"));

		try {
			JobProfile jobProfile = loadISJobConfiguration(jobName, bbJob.getParameters());

			validateJobParams(handler, bbJob, jobName, jobProfile);

			JobConf jobConf = prepareJob(getConf(clusterName), jobName, jobProfile, bbJob.getParameters());

			if (!hadoopClientMap.isMapreduceAvailable(clusterName))
				throw new HadoopServiceException("mapreduce not available for cluster: " + clusterName.toString());

			logJobDetails(jobConf);

			final RunningJob runningJob = hadoopClientMap.getJtClient(clusterName).submitJob(jobConf);
			final String jobId = newJobId(clusterName, runningJob.getID().getId());

			jobRegistry.registerJob(HadoopJob.newInstance(jobId, clusterName, jobProfile,
					new MapreduceJobMonitor(runningJob, newCompletionCallback(handler, bbJob, jobId))));

			updateJobStatus(jobName);
			handler.ongoing(bbJob);
		} catch (Throwable e) {
			log.error("error executing hadoop job: " + jobName, e);
			handler.failed(bbJob, e);
		}
	}

	protected JobConf prepareJob(final Configuration configuration, final String jobName, final JobProfile jobProfile, final Map<String, String> parameters)
			throws IOException, HadoopServiceException {

		log.info("creating job: " + jobName);

		JobConf jobConf = new JobConf(configuration);
		jobConf.setJobName(jobName);
		jobConf.set("dnet.mapred.job.description", jobProfile.getDescription());

		String jobLib = getJobLib(configuration, jobProfile);
		jobConf.setJar(new Path(jobLib).toString());

		set(jobConf, jobProfile.getJobDefinition());
		set(jobConf, parameters);

		// if we're reading from hbase table then we need also to set a scanner.
		ScanProperties scanProperties = jobProfile.getScanProperties();
		if (jobProfile.getRequiredParams().contains("hbase.mapreduce.inputtable") && (scanProperties != null)) {
			jobConf.set("hbase.mapreduce.scan", ScanFactory.getScan(scanProperties));
		}

		return jobConf;
	}

	protected String getJobLib(final Configuration configuration, final JobProfile jobProfile) throws HadoopServiceException {
		String jobLib = getDefaultLibPath(configuration.get("fs.defaultFS"));

		if (jobProfile.getJobDefinition().containsKey("job.lib")) {
			jobLib = jobProfile.getJobDefinition().get("job.lib");
		}

		if ((jobLib == null) || jobLib.isEmpty()) throw new HadoopServiceException("job.lib must refer to an absolute or relative HDFS path");
		if (!jobLib.startsWith("hdfs://")) {
			jobLib = configuration.get("fs.defaultFS") + jobLib;
		}

		log.info("using job.lib: " + jobLib);
		return jobLib;
	}

	protected void set(final JobConf jobConf, final Map<String, String> properties) {
		for (Entry<String, String> e : properties.entrySet()) {
			if (checkHdfsProperty(e)) {
				String v = jobConf.get("fs.defaultFS") + e.getValue();
				e.setValue(v);
			}
			jobConf.set(e.getKey(), e.getValue());
		}
	}

	protected void logJobDetails(final JobConf jobConf) {
		StringWriter sw = new StringWriter();
		try {
			jobConf.writeXml(sw);
			if (log.isDebugEnabled()) {
				log.debug("\n" + IndentXmlString.apply(sw.toString()));
			}
		} catch (IOException e) {
			log.warn("unable to log job details: " + jobConf.getJobName());
		}
	}

}
