package eu.dnetlib.data.mapreduce.hbase.dataimport;

import java.io.IOException;

import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Mapper;

import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.data.transform.Column;
import eu.dnetlib.data.transform.Row;
import eu.dnetlib.data.transform.XsltRowTransformer;
import eu.dnetlib.data.transform.XsltRowTransformerFactory;

public class ImportRecordsMapper extends Mapper<Text, Text, ImmutableBytesWritable, Put> {

	private static final boolean WRITE_TO_WAL = false;

	private XsltRowTransformer transformer;

	private String xslt;

	private ImmutableBytesWritable ibw;

	@Override
	protected void setup(Context context) throws IOException, InterruptedException {
		super.setup(context);

		this.xslt = context.getConfiguration().get(JobParams.HBASE_IMPORT_XSLT).trim();

		if (xslt == null || xslt.isEmpty()) { throw new IllegalArgumentException("missing xslt"); }

		transformer = new XsltRowTransformerFactory().getTransformer(xslt);
		ibw = new ImmutableBytesWritable();

		System.out.println("using trasformer: '" + transformer.getTransformerClassName() + "'");
	}

	@Override
	protected void map(Text key, Text value, Context context) throws IOException, InterruptedException {

		for (Row row : transformer.apply(value.toString())) {

			final byte[] rowKey = Bytes.toBytes(row.getKey());
			final Put put = new Put(rowKey);
			put.setWriteToWAL(WRITE_TO_WAL);

			for (Column<String, byte[]> col : row) {
				byte[] family = Bytes.toBytes(row.getColumnFamily());
				byte[] qualifier = Bytes.toBytes(col.getName());
				put.add(family, qualifier, col.getValue());
			}
			ibw.set(rowKey);
			context.write(ibw, put);
			context.getCounter("mdstore", row.getColumnFamily()).increment(row.getColumns().size());
		}
	}

}
