package eu.dnetlib.data.mapreduce.util;

import java.util.Arrays;
import java.util.Set;

import org.apache.hadoop.hbase.io.ImmutableBytesWritable;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.protobuf.Descriptors.Descriptor;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.InvalidProtocolBufferException;

import eu.dnetlib.data.proto.FieldTypeProtos.StructuredProperty;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.TypeProtos.Type;

public class OafUtils {

	public static Set<String> entities() {
		return Sets.newHashSet(Iterables.transform(Lists.newArrayList(Type.values()), new Function<Type, String>() {

			@Override
			public String apply(final Type t) {
				return t.toString();
			}
		}));
	}

	public static Predicate<Oaf> relationFilter() {
		return new Predicate<Oaf>() {

			@Override
			public boolean apply(Oaf oaf) {
				return oaf.getKind().equals(Kind.relation);
			}
		};
	}

	public static Predicate<Oaf> entityFilter() {
		return new Predicate<Oaf>() {

			@Override
			public boolean apply(Oaf oaf) {
				return oaf.getKind().equals(Kind.entity);
			}
		};
	}

	public static Function<OafDecoder, String> idDecoder() {
		return new Function<OafDecoder, String>() {

			@Override
			public String apply(OafDecoder input) {
				return input.getEntityId();
			}
		};
	}

	public static OafDecoder decode(final ImmutableBytesWritable oaf) {
		return new OafDecoder(oaf.copyBytes());
	}

	public static Function<ImmutableBytesWritable, OafDecoder> decoder() {
		return new Function<ImmutableBytesWritable, OafDecoder>() {

			@Override
			public OafDecoder apply(ImmutableBytesWritable input) {
				return OafDecoder.decode(input.copyBytes());
			}
		};
	}

	public static Function<ImmutableBytesWritable, Oaf> oafDecoder() {
		return new Function<ImmutableBytesWritable, Oaf>() {

			@Override
			public Oaf apply(ImmutableBytesWritable input) {
				return parse(input);
			}
		};
	}

	public static Oaf parse(ImmutableBytesWritable input) {
		try {
			return Oaf.parseFrom(input.copyBytes());
		} catch (InvalidProtocolBufferException e) {
			throw new IllegalArgumentException(e);
		}
	}

	public static Predicate<StructuredProperty> mainTitleFilter() {
		return new Predicate<StructuredProperty>() {

			@Override
			public boolean apply(StructuredProperty sp) {
				return sp.getQualifier() != null && sp.getQualifier().getClassname().equals("main title");
			}
		};
	}

	public static Set<String> getFieldNames(final Descriptor d, Integer... tag) {
		return Sets.newHashSet(Iterables.transform(Arrays.asList(tag), new Function<Integer, String>() {

			@Override
			public String apply(Integer i) {
				FieldDescriptor fd = d.findFieldByNumber(i);
				if (fd == null) throw new IllegalArgumentException("undefined tag: " + i + " for type: " + d.getFullName());
				return fd.getName();
			}
		}));
	}

}
