package eu.dnetlib.data.mapreduce.hbase.index;

import java.io.IOException;
import java.util.List;
import java.util.Map.Entry;

import org.apache.commons.codec.binary.Base64;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.CloudSolrServer;
import org.apache.solr.client.solrj.response.UpdateResponse;
import org.apache.solr.common.SolrInputDocument;

import com.google.common.collect.Lists;

import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.functionality.index.solr.feed.InputDocumentFactory;
import eu.dnetlib.functionality.index.solr.feed.StreamingInputDocumentFactory;
import eu.dnetlib.miscutils.datetime.HumanTime;
import eu.dnetlib.miscutils.functional.xml.ApplyXslt;

public class IndexFeedMapper extends Mapper<Text, Text, Text, Text> {

	private InputDocumentFactory documentFactory;

	private CloudSolrServer solrServer;

	private String version;

	private String dsId;

	private int shutdownWaitTime = 10000;

	private int bufferFlushThreshold = 100;

	private ApplyXslt dmfToRecord;

	private List<SolrInputDocument> buffer;

	private boolean simulation = false;

	@Override
	protected void setup(final Context context) throws IOException, InterruptedException {

		logConfiguration(context.getConfiguration());

		String baseURL = context.getConfiguration().get(JobParams.INDEX_SOLR_URL);
		System.out.println("solr server baseURL: " + baseURL);

		String collection = context.getConfiguration().get(JobParams.INDEX_SOLR_COLLECTION);
		System.out.println("solr server collection: " + collection);

		solrServer = new CloudSolrServer(baseURL);
		solrServer.setParallelUpdates(true);
		solrServer.setDefaultCollection(collection);

		dsId = context.getConfiguration().get(JobParams.INDEX_DSID);
		shutdownWaitTime = Integer.parseInt(context.getConfiguration().get(JobParams.INDEX_SHUTDOWN_WAIT));
		bufferFlushThreshold = Integer.parseInt(context.getConfiguration().get(JobParams.INDEX_BUFFER_FLUSH_TRESHOLD));
		documentFactory = new StreamingInputDocumentFactory();
		version = InputDocumentFactory.getParsedDateField(context.getConfiguration().get(JobParams.INDEX_FEED_TIME));
		buffer = Lists.newArrayList();
		simulation = Boolean.parseBoolean(context.getConfiguration().get(JobParams.INDEX_FEED_SIMULATION_MODE));

		final String xslt = new String(Base64.decodeBase64(context.getConfiguration().get(JobParams.INDEX_XSLT)));

		System.out.println("got xslt: \n" + xslt);
		System.out.println("got version: " + version);
		System.out.println("simulation: " + simulation);
		System.out.println("buffer size: " + bufferFlushThreshold);

		dmfToRecord = new ApplyXslt(xslt);
	}

	@Override
	protected void map(final Text key, final Text value, final Context context) throws IOException, InterruptedException {

		String indexRecord = "";
		SolrInputDocument doc = null;

		// if (!value.toString().contains("oaf:person")) {
		// return;
		// }

		try {
			indexRecord = dmfToRecord.evaluate(value.toString());
			doc = documentFactory.parseDocument(version, indexRecord, dsId, "dnetResult");

			if (!doc.isEmpty()) {

				buffer.add(doc);
				if (buffer.size() >= bufferFlushThreshold) {
					doAdd(buffer, context);
					// Thread.sleep(100);
				}
			} else {
				context.getCounter("index", "skipped records").increment(1);
			}
		} catch (Throwable e) {
			context.getCounter("index", e.getClass().toString()).increment(1);
			context.write(key, printRottenRecord(context.getTaskAttemptID().toString(), value, indexRecord, doc));
			e.printStackTrace(System.err);
		}
	}

	private void doAdd(final List<SolrInputDocument> buffer, final Context context) throws SolrServerException, IOException {
		if (!simulation) {
			long start = System.currentTimeMillis();
			UpdateResponse rsp = solrServer.add(buffer);
			long stop = System.currentTimeMillis() - start;
			System.out.println("feed time for " + buffer.size() + " records : " + HumanTime.exactly(stop) + "\n");
			context.getCounter("index", "status code: " + rsp.getStatus()).increment(buffer.size());
		}
		buffer.clear();
	}

	@Override
	protected void cleanup(final Context context) throws IOException, InterruptedException {
		super.cleanup(context);
		try {
			if (!buffer.isEmpty()) {
				doAdd(buffer, context);
			}
			System.out.println("\nwaiting " + shutdownWaitTime + "ms before shutdown");
			Thread.sleep(shutdownWaitTime);
			solrServer.shutdown();
		} catch (SolrServerException e) {
			System.err.println("couldn't shutdown server " + e.getMessage());
		}
	}

	private Text printRottenRecord(final String taskid, final Text value, final String indexRecord, final SolrInputDocument doc) {
		return new Text("\n**********************************\n" + "task: " + taskid + "\n"
				+ check("original", value.toString() + check("indexRecord", indexRecord) + check("solrDoc", doc)));
	}

	private String check(final String label, final Object value) {
		if ((value != null) && !value.toString().isEmpty()) return "\n " + label + ":\n" + value + "\n";
		return "\n";
	}

	private void logConfiguration(final Configuration conf) {
		System.out.println("job configutation #################");
		for (Entry<String, String> e : conf) {
			System.out.println("'" + e.getKey() + "' : '" + e.getValue() + "'");
		}
		System.out.println("end of job configutation #################\n\n");
	}

}
