package eu.dnetlib.data.mapreduce.hbase.index;

import com.google.common.collect.Lists;
import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.functionality.index.solr.feed.InputDocumentFactory;
import eu.dnetlib.functionality.index.solr.feed.StreamingInputDocumentFactory;
import eu.dnetlib.miscutils.datetime.HumanTime;
import eu.dnetlib.miscutils.functional.xml.ApplyXslt;
import org.apache.commons.codec.binary.Base64;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.CloudSolrServer;
import org.apache.solr.client.solrj.response.SolrPingResponse;
import org.apache.solr.client.solrj.response.UpdateResponse;
import org.apache.solr.common.SolrInputDocument;
import org.apache.solr.common.util.NamedList;
import org.apache.solr.common.util.SimpleOrderedMap;
import org.hsqldb.Trace;

import javax.xml.stream.XMLStreamException;
import java.io.IOException;
import java.util.List;
import java.util.Map.Entry;

public class IndexFeedMapper extends Mapper<Text, Text, Text, Text> {

    private InputDocumentFactory documentFactory;

    private CloudSolrServer solrServer;

    private String version;

    private String dsId;

    private int shutdownWaitTime = 10000;

    private int bufferFlushThreshold = 100;

    private ApplyXslt dmfToRecord;

    private List<SolrInputDocument> buffer;

    private int backoffTimeMs = 5000;

    private boolean simulation = false;

    @Override
    protected void setup(final Context context) throws IOException, InterruptedException {

        logConfiguration(context.getConfiguration());

        dsId = context.getConfiguration().get(JobParams.INDEX_DSID);
        shutdownWaitTime = Integer.parseInt(context.getConfiguration().get(JobParams.INDEX_SHUTDOWN_WAIT));
        bufferFlushThreshold = Integer.parseInt(context.getConfiguration().get(JobParams.INDEX_BUFFER_FLUSH_TRESHOLD));
        documentFactory = new StreamingInputDocumentFactory();
        version = InputDocumentFactory.getParsedDateField(context.getConfiguration().get(JobParams.INDEX_FEED_TIME));
        buffer = Lists.newArrayList();
        simulation = Boolean.parseBoolean(context.getConfiguration().get(JobParams.INDEX_FEED_SIMULATION_MODE));

        final String xslt = new String(Base64.decodeBase64(context.getConfiguration().get(JobParams.INDEX_XSLT)));

        System.out.println("got xslt: \n" + xslt);
        System.out.println("got version: " + version);
        System.out.println("simulation: " + simulation);
        System.out.println("buffer size: " + bufferFlushThreshold);

        dmfToRecord = new ApplyXslt(xslt);


        String baseURL = context.getConfiguration().get(JobParams.INDEX_SOLR_URL);
        System.out.println("solr server baseURL: " + baseURL);

        String collection = context.getConfiguration().get(JobParams.INDEX_SOLR_COLLECTION);
        System.out.println("solr server collection: " + collection);

        while(true) {
            try {
                System.out.println("initializing solr server...");
                solrServer = new CloudSolrServer(baseURL);

                solrServer.connect();

                solrServer.setParallelUpdates(true);
                solrServer.setDefaultCollection(collection);

                SolrPingResponse rsp = solrServer.ping();

                if (rsp.getStatus() != 0) {
                    throw new SolrServerException("bad init status: " + rsp.getStatus());
                } else break;

            } catch (Throwable e) {
                if (solrServer != null) {
                    solrServer.shutdown();
                }
                context.getCounter("index init", e.getMessage()).increment(1);
                System.out.println(String.format("failed to init solr client wait %dms", backoffTimeMs));
                Thread.sleep(backoffTimeMs);
            }
        }
    }

    @Override
    protected void map(final Text key, final Text value, final Context context) throws IOException, InterruptedException {

        String indexRecord = "";
        SolrInputDocument doc = null;

        try {
            indexRecord = dmfToRecord.evaluate(value.toString());
            doc = documentFactory.parseDocument(version, indexRecord, dsId, "dnetResult");
        } catch (XMLStreamException e) {
            handleError(key, value, context, indexRecord, doc, e);
        }

        while (true) {
            try {
                addDocument(context, doc);
                return;
            } catch (Throwable e) {
                context.getCounter("index feed", "retries").increment(1);
                handleError(key, value, context, indexRecord, doc, e);
                System.out.println(String.format("failed to feed documents, waiting %dms", backoffTimeMs));
                Thread.sleep(backoffTimeMs);
            }
        }
    }

    private void addDocument(Context context, SolrInputDocument doc) throws SolrServerException, IOException {
        if (!doc.isEmpty()) {

            buffer.add(doc);
            if (buffer.size() >= bufferFlushThreshold) {
                doAdd(buffer, context);
                // Thread.sleep(100);
            }
        } else {
            context.getCounter("index feed", "skipped records").increment(1);
        }
    }

    private void doAdd(final List<SolrInputDocument> buffer, final Context context) throws SolrServerException, IOException {
        if (!simulation) {
            long start = System.currentTimeMillis();
            UpdateResponse rsp = solrServer.add(buffer);
            long stop = System.currentTimeMillis() - start;
            System.out.println("feed time for " + buffer.size() + " records : " + HumanTime.exactly(stop) + "\n");

            int status = rsp.getStatus();
            context.getCounter("index feed", "status code: " + status).increment(buffer.size());
            
            if (status != 0) {
                throw new SolrServerException("bad status: " + status);
            }
        }
        buffer.clear();
    }

    @Override
    protected void cleanup(final Context context) throws IOException, InterruptedException {
        super.cleanup(context);
        try {
            if (!buffer.isEmpty()) {
                doAdd(buffer, context);
            }
            System.out.println("\nwaiting " + shutdownWaitTime + "ms before shutdown");
            Thread.sleep(shutdownWaitTime);
            solrServer.shutdown();
        } catch (SolrServerException e) {
            System.err.println("couldn't shutdown server " + e.getMessage());
        }
    }

    private void handleError(Text key, Text value, Context context, String indexRecord, SolrInputDocument doc, Throwable e) throws IOException, InterruptedException {
        context.getCounter("index feed", e.getClass().getName()).increment(1);
        context.write(key, printRottenRecord(context.getTaskAttemptID().toString(), value, indexRecord, doc));
        e.printStackTrace(System.err);
    }


    private Text printRottenRecord(final String taskid, final Text value, final String indexRecord, final SolrInputDocument doc) {
        return new Text("\n**********************************\n" + "task: " + taskid + "\n"
                + check("original", value.toString() + check("indexRecord", indexRecord) + check("solrDoc", doc)));
    }

    private String check(final String label, final Object value) {
        if ((value != null) && !value.toString().isEmpty()) return "\n " + label + ":\n" + value + "\n";
        return "\n";
    }

    private void logConfiguration(final Configuration conf) {
        System.out.println("job configutation #################");
        for (Entry<String, String> e : conf) {
            System.out.println("'" + e.getKey() + "' : '" + e.getValue() + "'");
        }
        System.out.println("end of job configutation #################\n\n");
    }

}
