package eu.dnetlib.data.mapreduce.util;

import java.nio.ByteBuffer;

import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.util.Bytes;

import eu.dnetlib.data.proto.DedupProtos.Dedup;
import eu.dnetlib.data.proto.DedupSimilarityProtos.DedupSimilarity;
import eu.dnetlib.data.proto.FieldTypeProtos.Qualifier;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.OafProtos.OafRel.Builder;
import eu.dnetlib.data.proto.OrganizationOrganizationProtos.OrganizationOrganization;
import eu.dnetlib.data.proto.PersonPersonProtos.PersonPerson;
import eu.dnetlib.data.proto.RelMetadataProtos.RelMetadata;
import eu.dnetlib.data.proto.RelTypeProtos.RelType;
import eu.dnetlib.data.proto.RelTypeProtos.SubRelType;
import eu.dnetlib.data.proto.ResultResultProtos.ResultResult;
import eu.dnetlib.data.proto.TypeProtos.Type;
import eu.dnetlib.pace.util.DedupConfig;

public class DedupUtils {

	public static final String CF_SEPARATOR = "_";

	public static final String ROOT = "dedup_wf";

	public static final String BODY_S = "body";

	public static final byte[] BODY_B = Bytes.toBytes(BODY_S);

	public static String dedupPrefix(final String dedupRun) {
		return "|" + ROOT + "_" + dedupRun + "::";
	}

	public static String newId(final String id, final String dedupRun) {
		if ((dedupRun == null) || (dedupRun.length() != 3)) throw new IllegalArgumentException("wrong dedupRun param");

		return id.replaceFirst("\\|.*\\:\\:", dedupPrefix(dedupRun));
	}

	public static byte[] newIdBytes(final String s, final String dedupRun) {
		return newId(s, dedupRun).getBytes();
	}

	public static byte[] newIdBytes(final ByteBuffer b, final String dedupRun) {
		return newId(new String(b.array()), dedupRun).getBytes();
	}

	public static boolean isRoot(final String s) {
		return s.contains(ROOT);
	}

	public static boolean isRoot(final ImmutableBytesWritable s) {
		return isRoot(s.copyBytes());
	}

	public static boolean isRoot(final byte[] s) {
		return isRoot(new String(s));
	}

	public static String getDedupCF_merges(final Type type) {
		return getRelType(type) + CF_SEPARATOR + SubRelType.dedup + CF_SEPARATOR + Dedup.RelName.merges;
	}

	public static byte[] getDedupCF_mergesBytes(final Type type) {
		return Bytes.toBytes(getDedupCF_merges(type));
	}

	public static String getDedupCF_mergedIn(final Type type) {
		return getRelType(type) + CF_SEPARATOR + SubRelType.dedup + CF_SEPARATOR + Dedup.RelName.isMergedIn;
	}

	public static byte[] getDedupCF_mergedInBytes(final Type type) {
		return Bytes.toBytes(getDedupCF_mergedIn(type));
	}

	public static String getSimilarityCF(final Type type) {
		return getRelType(type) + CF_SEPARATOR + SubRelType.dedupSimilarity + CF_SEPARATOR + DedupSimilarity.RelName.isSimilarTo;
	}

	public static byte[] getSimilarityCFBytes(final Type type) {
		return Bytes.toBytes(getSimilarityCF(type));
	}

	public static String getRelTypeString(final Type type) {
		return getRelType(type).toString();
	}

	public static RelType getRelType(final Type type) {
		switch (type) {
		case organization:
			return RelType.organizationOrganization;
		case person:
			return RelType.personPerson;
		case result:
			return RelType.resultResult;
		default:
			throw new IllegalArgumentException("Deduplication not supported for entity type: " + type);
		}
	}

	public static ColumnFamily decodeCF(final byte[] b) {
		String[] s = new String(b).split(CF_SEPARATOR);
		return new DedupUtils().getCF(RelType.valueOf(s[0]), SubRelType.valueOf(s[1]));
	}

	private ColumnFamily getCF(final RelType relType, final SubRelType subRelType) {
		return new ColumnFamily(relType, subRelType);
	}

	public static OafRel.Builder getDedup(final DedupConfig dedupConf, final String from, final String to, final Dedup.RelName relClass) {
		Type type = Type.valueOf(dedupConf.getEntityType());
		RelType relType = DedupUtils.getRelType(type);
		Builder oafRel =
				OafRel.newBuilder().setRelType(relType).setSubRelType(SubRelType.dedup).setRelClass(relClass.toString()).setChild(false)
						.setSource(new String(from)).setTarget(new String(to));
		switch (type) {
		case organization:
			oafRel.setOrganizationOrganization(OrganizationOrganization.newBuilder().setDedup(
					DedupUtils.dedup(relClass, "dnet:organization_organization_relations")));
			break;
		case person:
			oafRel.setPersonPerson(PersonPerson.newBuilder().setDedup(DedupUtils.dedup(relClass, "dnet:person_person_relations")));
			break;
		case result:
			oafRel.setResultResult(ResultResult.newBuilder().setDedup(DedupUtils.dedup(relClass, "dnet:result_result_relations")));
			break;
		default:
			throw new IllegalArgumentException("Deduplication not supported for entity type: " + dedupConf.getEntityType());
		}
		return oafRel;
	}

	private static Dedup.Builder dedup(final Dedup.RelName relClass, final String scheme) {
		return Dedup.newBuilder().setRelMetadata(
				RelMetadata.newBuilder().setSemantics(
						Qualifier.newBuilder().setClassid(relClass.toString()).setClassname(relClass.toString()).setSchemeid(scheme).setSchemename(scheme)));
	}

	class ColumnFamily {

		private final RelType relType;
		private final SubRelType subRelType;

		public ColumnFamily(final RelType relType, final SubRelType subRelType) {
			this.relType = relType;
			this.subRelType = subRelType;
		}

		@Override
		public String toString() {
			return getRelType() + CF_SEPARATOR + getSubRelType();
		}

		public RelType getRelType() {
			return relType;
		}

		public SubRelType getSubRelType() {
			return subRelType;
		}

	}

}
