package eu.dnetlib.data.mapreduce.util;

import static eu.dnetlib.miscutils.collections.MappedCollection.listMap;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.StringTokenizer;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.commons.lang.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.io.SAXReader;

import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.google.protobuf.Descriptors.EnumValueDescriptor;
import com.google.protobuf.Descriptors.FieldDescriptor;
import com.google.protobuf.GeneratedMessage;
import com.mycila.xmltool.XMLDoc;
import com.mycila.xmltool.XMLTag;

import eu.dnetlib.data.mapreduce.hbase.index.config.ContextDef;
import eu.dnetlib.data.mapreduce.hbase.index.config.ContextMapper;
import eu.dnetlib.data.mapreduce.hbase.index.config.EntityConfigTable;
import eu.dnetlib.data.mapreduce.hbase.index.config.LinkDescriptor;
import eu.dnetlib.data.mapreduce.hbase.index.config.RelClasses;
import eu.dnetlib.data.proto.FieldTypeProtos.DataInfo;
import eu.dnetlib.data.proto.FieldTypeProtos.ExtraInfo;
import eu.dnetlib.data.proto.FieldTypeProtos.KeyValue;
import eu.dnetlib.data.proto.FieldTypeProtos.Qualifier;
import eu.dnetlib.data.proto.FieldTypeProtos.StringField;
import eu.dnetlib.data.proto.FieldTypeProtos.StructuredProperty;
import eu.dnetlib.data.proto.OafProtos.OafEntity;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.PersonProtos.Person;
import eu.dnetlib.data.proto.ProjectProtos.Project;
import eu.dnetlib.data.proto.RelMetadataProtos.RelMetadata;
import eu.dnetlib.data.proto.ResultProtos.Result;
import eu.dnetlib.data.proto.ResultProtos.Result.Context;
import eu.dnetlib.data.proto.ResultProtos.Result.ExternalReference;
import eu.dnetlib.data.proto.ResultProtos.Result.Instance;
import eu.dnetlib.data.proto.ResultProtos.Result.Journal;
import eu.dnetlib.data.proto.TypeProtos.Type;
import eu.dnetlib.miscutils.functional.UnaryFunction;

public class XmlRecordFactory {

	protected Set<String> specialDatasourceTypes = Sets.newHashSet("scholarcomminfra", "infospace", "pubsrepository::mock", "entityregistry");

	protected TemplateFactory templateFactory = new TemplateFactory();

	protected OafDecoder mainEntity = null;

	protected String key = null;

	protected List<OafDecoder> relations = Lists.newLinkedList();
	protected List<OafDecoder> children = Lists.newLinkedList();

	protected EntityConfigTable entityConfigTable;

	protected ContextMapper contextMapper;

	protected RelClasses relClasses;

	protected String schemaLocation;

	protected boolean entityDefaults;
	protected boolean relDefaults;
	protected boolean childDefaults;

	protected Set<String> contextes = Sets.newHashSet();

	protected List<String> extraInfo = Lists.newArrayList();

	protected Map<String, Integer> counters = Maps.newHashMap();

	protected Transformer transformer;

	public XmlRecordFactory(final EntityConfigTable entityConfigTable, final ContextMapper contextMapper, final RelClasses relClasses,
			final String schemaLocation, final boolean entityDefaults, final boolean relDefaults, final boolean childDefeaults)
			throws TransformerConfigurationException, TransformerFactoryConfigurationError {
		this.entityConfigTable = entityConfigTable;
		this.contextMapper = contextMapper;
		this.relClasses = relClasses;
		this.schemaLocation = schemaLocation;
		this.entityDefaults = entityDefaults;
		this.relDefaults = relDefaults;
		this.childDefaults = childDefeaults;

		transformer = TransformerFactory.newInstance().newTransformer();
		transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
	}

	public String getId() {
		return key;
	}

	public boolean isValid() {
		return mainEntity != null;
	}

	public void setMainEntity(final OafDecoder mainEntity) {
		this.mainEntity = mainEntity;
		this.key = mainEntity.decodeEntity().getId();
	}

	public void addRelation(final OafDecoder rel) {
		addRelOrChild(relations, rel);
	}

	public void addChild(final OafDecoder child) {
		addRelOrChild(children, child);
	}

	private void addRelOrChild(final List<OafDecoder> list, final OafDecoder decoder) {
		list.add(decoder);
	}

	public String build() {

		OafEntityDecoder entity = mainEntity.decodeEntity();
		// System.out.println("building");
		// System.out.println("main: " + mainEntity);
		// System.out.println("rel:  " + relations);
		// System.out.println("chi:  " + children);
		// System.out.println("=============");

		final Type type = entity.getType();
		final List<String> metadata = decodeType(entity, null, entityDefaults, false);

		// rels has to be processed before the contexts because they enrich the contextMap with the funding info.
		List<String> rels = listRelations();
		metadata.addAll(buildContexts(type));
		metadata.add(parseDataInfo(mainEntity));

		final String body = templateFactory.buildBody(type, metadata, rels, listChildren(), extraInfo);

		// System.out.println("record id: " + recordId);
		return templateFactory.buildRecord(type, key, entity.getDateOfCollection(), schemaLocation, body, countersAsXml());
	}

	private String parseDataInfo(final OafDecoder decoder) {
		DataInfo dataInfo = decoder.getOaf().getDataInfo();

		StringBuilder sb = new StringBuilder();
		sb.append("<datainfo>");
		sb.append(asXmlElement("inferred", dataInfo.getInferred() + "", null, null));
		sb.append(asXmlElement("deletedbyinference", dataInfo.getDeletedbyinference() + "", null, null));
		sb.append(asXmlElement("trust", dataInfo.getTrust() + "", null, null));
		sb.append(asXmlElement("inferenceprovenance", dataInfo.getInferenceprovenance() + "", null, null));
		sb.append(asXmlElement("provenanceaction", null, dataInfo.getProvenanceaction(), null));
		sb.append("</datainfo>");

		return sb.toString();
	}

	private List<String> decodeType(final OafEntityDecoder decoder, final Set<String> filter, final boolean defaults, final boolean expandingRel) {

		final List<String> metadata = Lists.newArrayList();
		metadata.addAll(listFields(decoder.getMetadata(), filter, defaults, expandingRel));
		metadata.addAll(listFields(decoder.getOafEntity(), filter, defaults, expandingRel));

		if (decoder.getEntity() instanceof Result && !expandingRel) {
			metadata.add(asXmlElement("bestlicense", "", getBestLicense(), null));

			metadata.addAll(listFields(decoder.getEntity(), filter, defaults, expandingRel));
		}
		if (decoder.getEntity() instanceof Person && !expandingRel) {
			metadata.addAll(listFields(decoder.getEntity(), filter, defaults, expandingRel));
		}
		if (decoder.getEntity() instanceof Project && !expandingRel) {
			metadata.addAll(listFields(decoder.getEntity(), filter, defaults, expandingRel));
		}

		return metadata;
	}

	private Qualifier getBestLicense() {
		Qualifier bestLicense = getQualifier("UNKNOWN", "not available", "dnet:access_modes");
		LicenseComparator lc = new LicenseComparator();
		for (Instance instance : ((Result) mainEntity.decodeEntity().getEntity()).getInstanceList()) {
			if (lc.compare(bestLicense, instance.getLicence()) > 0) {
				bestLicense = instance.getLicence();
			}
		}
		return bestLicense;
	}

	public Qualifier getQualifier(final String classid, final String classname, final String schemename) {
		return Qualifier.newBuilder().setClassid(classid).setClassname(classname).setSchemeid(schemename).setSchemename(schemename).build();
	}

	private List<String> listRelations() {

		final List<String> rels = Lists.newArrayList();

		for (OafDecoder decoder : this.relations) {

			final OafRel rel = decoder.getOafRel();
			final OafEntity cachedTarget = rel.getCachedTarget();
			final OafRelDecoder relDecoder = OafRelDecoder.decode(rel);

			// if (!relDecoder.getRelType().equals(RelType.personResult) || relDecoder.getRelTargetId().equals(key)) {
			if (relDecoder.getRelSourceId().equals(key) || relDecoder.getRelTargetId().equals(key)) {

				final List<String> metadata = Lists.newArrayList();
				Type targetType = relDecoder.getTargetType(mainEntity.getEntity().getType());
				Set<String> relFilter = entityConfigTable.getFilter(targetType, relDecoder.getRelDescriptor());
				metadata.addAll(listFields(relDecoder.getSubRel(), relFilter, false, true));

				String semanticclass = "";
				String semanticscheme = "";

				RelDescriptor relDescriptor = relDecoder.getRelDescriptor();

				if (cachedTarget != null && cachedTarget.isInitialized()) {

					final Set<String> filter = entityConfigTable.getFilter(targetType, relDescriptor);
					metadata.addAll(decodeType(OafEntityDecoder.decode(cachedTarget), filter, relDefaults, true));
				}

				RelMetadata relMetadata = relDecoder.getRelMetadata();
				// debug
				if (relMetadata == null) {
					// System.err.println(this);
					semanticclass = semanticscheme = "UNKNOWN";
				} else {
					semanticclass = relClasses.getInverse(relMetadata.getSemantics().getClassname());
					semanticscheme = relMetadata.getSemantics().getSchemename();
				}

				incrementCounter(relDescriptor.getSubRelType().toString());

				LinkDescriptor ld = entityConfigTable.getDescriptor(relDecoder.getTargetType(mainEntity.getEntity().getType()), relDescriptor);

				String relId = ld != null && !ld.isSymmetric() ? relDecoder.getRelTargetId() : relDecoder.getRelSourceId();

				DataInfo info = decoder.getOaf().getDataInfo();

				rels.add(templateFactory.getRel(targetType, relId, metadata, semanticclass, semanticscheme, info.getInferred(), info.getTrust(),
						info.getInferenceprovenance(), info.getProvenanceaction().getClassid()));
			}
		}
		return rels;
	}

	private List<String> listChildren() {

		final List<String> children = Lists.newArrayList();
		for (OafDecoder decoder : this.children) {
			OafEntity cachedTarget = decoder.getOafRel().getCachedTarget();
			addChildren(children, cachedTarget, decoder.getRelDescriptor());
		}
		OafEntityDecoder entity = mainEntity.decodeEntity();
		if (entity.getType().equals(Type.result)) {
			for (Instance instance : ((Result) entity.getEntity()).getInstanceList()) {
				children.add(templateFactory.getInstance(instance.getHostedby().getKey(), listFields(instance, null, false, false),
						listMap(instance.getUrlList(), new UnaryFunction<String, String>() {

							@Override
							public String evaluate(final String identifier) {
								return templateFactory.getWebResource(identifier);
							}
						})));
			}
			for (ExternalReference er : ((Result) entity.getEntity()).getExternalReferenceList()) {
				// Set<String> filters = entityConfigTable.getFilter(Type.result, RelType.resultResult);
				List<String> fields = listFields(er, null, false, false);
				children.add(templateFactory.getChild("externalreference", null, fields));
			}
		}

		return children;
	}

	private void addChildren(final List<String> children, final OafEntity target, final RelDescriptor relDescriptor) {
		final OafEntityDecoder decoder = OafEntityDecoder.decode(target);
		incrementCounter(relDescriptor.getSubRelType().toString());
		Set<String> filters = entityConfigTable.getFilter(target.getType(), relDescriptor);
		children.add(templateFactory.getChild(decoder.getType().toString(), decoder.getId(), listFields(decoder.getMetadata(), filters, childDefaults, false)));
	}

	// //////////////////////////////////

	private List<String> listFields(final GeneratedMessage fields, final Set<String> filter, final boolean defaults, final boolean expandingRel) {

		final List<String> metadata = Lists.newArrayList();

		if (fields != null) {

			Set<String> seen = Sets.newHashSet();
			for (Entry<FieldDescriptor, Object> e : filterFields(fields, filter)) {

				// final String name = getFieldName(e.getKey().getName());
				final String name = e.getKey().getName();
				seen.add(name);

				addFieldValue(metadata, e.getKey(), e.getValue(), expandingRel);
			}

			if (defaults) {
				for (FieldDescriptor fd : fields.getDescriptorForType().getFields()) {
					if (!seen.contains(fd.getName())) {
						addFieldValue(metadata, fd, getDefault(fd), expandingRel);
					}
				}
			}
		}
		return metadata;
	}

	private Object getDefault(final FieldDescriptor fd) {
		switch (fd.getType()) {
		case BOOL:
			return false;
		case BYTES:
			return "".getBytes();
		case MESSAGE: {
			if (Qualifier.getDescriptor().equals(fd.getMessageType())) { return defaultQualifier(); }
			if (StructuredProperty.getDescriptor().equals(fd.getMessageType())) { return StructuredProperty.newBuilder().setValue("")
					.setQualifier(defaultQualifier()).build(); }
			if (KeyValue.getDescriptor().equals(fd.getMessageType())) { return KeyValue.newBuilder().setKey("").setValue("").build(); }
			if (StringField.getDescriptor().equals(fd.getMessageType())) { return StringField.newBuilder().setValue("").build(); }
			return null;
		}
		case SFIXED32:
		case SFIXED64:
		case SINT32:
		case SINT64:
		case INT32:
		case INT64:
		case DOUBLE:
		case FIXED32:
		case FIXED64:
		case FLOAT:
			return 0;
		case STRING:
			return "";
		default:
			return null;
		}
	}

	private Qualifier defaultQualifier() {
		return Qualifier.newBuilder().setClassid("").setClassname("").setSchemeid("").setSchemename("").build();
	}

	@SuppressWarnings("unchecked")
	private void addFieldValue(final List<String> metadata, final FieldDescriptor fd, final Object value, final boolean expandingRel) {
		if (fd.getName().equals("dateofcollection") || fd.getName().equals("id") || fd.getName().equals("url") || value == null) { return; }

		if (fd.getName().equals("datasourcetype")) {
			String classid = ((Qualifier) value).getClassid();

			Qualifier.Builder q = Qualifier.newBuilder((Qualifier) value);
			if (specialDatasourceTypes.contains(classid)) {
				q.setClassid("other").setClassname("other");
			}
			metadata.add(asXmlElement("datasourcetypeui", "", q.build(), null));
		}

		if (fd.isRepeated() && value instanceof List<?>) {
			for (Object o : (List<Object>) value) {
				guessType(metadata, fd, o, expandingRel);
			}
		} else {
			guessType(metadata, fd, value, expandingRel);
		}
	}

	private void guessType(final List<String> metadata, final FieldDescriptor fd, final Object o, final boolean expandingRel) {

		if (fd.getType().equals(FieldDescriptor.Type.MESSAGE)) {

			if (Qualifier.getDescriptor().equals(fd.getMessageType())) {
				Qualifier qualifier = (Qualifier) o;
				metadata.add(asXmlElement(fd.getName(), "", qualifier, null));
			}

			if (StructuredProperty.getDescriptor().equals(fd.getMessageType())) {
				StructuredProperty sp = (StructuredProperty) o;
				metadata.add(asXmlElement(fd.getName(), sp.getValue(), sp.getQualifier(), sp.hasDataInfo() ? sp.getDataInfo() : null));
			}

			if (KeyValue.getDescriptor().equals(fd.getMessageType())) {
				KeyValue kv = (KeyValue) o;
				metadata.add("<" + fd.getName() + " name=\"" + escapeXml(kv.getValue()) + "\" id=\"" + escapeXml(removePrefix(kv.getKey())) + "\"/>");
			}

			if (StringField.getDescriptor().equals(fd.getMessageType())) {
				String fieldName = fd.getName();

				if (fieldName.equals("fundingtree")) {
					String xmlTree = o instanceof StringField ? ((StringField) o).getValue() : o.toString();

					if (expandingRel) {
						metadata.add(getRelFundingTree(xmlTree));
						fillContextMap(xmlTree);
					} else {
						metadata.add(xmlTree);
					}
				} else {
					StringField sf = (StringField) o;
					StringBuilder sb = new StringBuilder("<" + fd.getName());
					if (sf.hasDataInfo()) {
						DataInfo dataInfo = sf.getDataInfo();
						dataInfoAsAttributes(sb, dataInfo);
					}
					sb.append(">" + escapeXml(sf.getValue()) + "</" + fd.getName() + ">");
					metadata.add(sb.toString());
				}
			}
			if (Journal.getDescriptor().equals(fd.getMessageType()) && o != null) {
				Journal j = (Journal) o;
				metadata.add("<journal " + "issn=\"" + escapeXml(j.getIssnPrinted()) + "\" " + "eissn=\"" + escapeXml(j.getIssnOnline()) + "\" " + "lissn=\""
						+ escapeXml(j.getIssnLinking()) + "\">" + escapeXml(j.getName()) + "</journal>");
			}

			if (Context.getDescriptor().equals(fd.getMessageType()) && o != null) {
				contextes.add(((Result.Context) o).getId());
			}

			if (ExtraInfo.getDescriptor().equals(fd.getMessageType()) && o != null) {

				ExtraInfo e = (ExtraInfo) o;
				StringBuilder sb = new StringBuilder("<" + fd.getName() + " ");

				sb.append("name=\"" + e.getName() + "\" ");
				sb.append("typology=\"" + e.getTypology() + "\" ");
				sb.append("provenance=\"" + e.getProvenance() + "\" ");
				sb.append("trust=\"" + e.getTrust() + "\"");
				sb.append(">");
				sb.append(e.getValue());
				sb.append("</" + fd.getName() + ">");

				extraInfo.add(sb.toString());
			}

		} else if (fd.getType().equals(FieldDescriptor.Type.ENUM)) {
			if (fd.getFullName().equals("eu.dnetlib.data.proto.OafEntity.type")) { return; }
			metadata.add(asXmlElement(fd.getName(), ((EnumValueDescriptor) o).getName(), null, null));
		} else {
			metadata.add(asXmlElement(fd.getName(), o.toString(), null, null));
		}
	}

	private StringBuilder dataInfoAsAttributes(final StringBuilder sb, final DataInfo dataInfo) {
		sb.append(" inferred=\"" + dataInfo.getInferred() + "\"");
		sb.append(" inferenceprovenance=\"" + dataInfo.getInferenceprovenance() + "\"");
		sb.append(" provenanceaction=\"" + dataInfo.getProvenanceaction().getClassid() + "\"");
		sb.append(" trust=\"" + dataInfo.getTrust() + "\" ");
		return sb;
	}

	private List<String> buildContexts(final Type type) {
		final List<String> res = Lists.newArrayList();

		if (contextMapper != null && !contextMapper.isEmpty() && type.equals(Type.result)) {

			XMLTag document = XMLDoc.newDocument(true).addRoot("contextRoot");

			for (String id : contextes) {

				StringTokenizer st = new StringTokenizer(id, "::");
				String token = "";
				while (st.hasMoreTokens()) {
					token += st.nextToken();

					final ContextDef def = contextMapper.get(token);

					if (def == null) { throw new IllegalStateException("cannot find context for id: " + token); }

					if (def.getName().equals("context")) {
						String xpath = "//context/@id='" + def.getId() + "'";
						if (!document.gotoRoot().rawXpathBoolean(xpath, new Object())) {
							document = addContextDef(document.gotoRoot(), def);
						}
					}

					if (def.getName().equals("category")) {
						String rootId = StringUtils.substringBefore(def.getId(), "::");
						document = addContextDef(document.gotoRoot().gotoTag("//context[./@id='" + rootId + "']", new Object()), def);
					}

					if (def.getName().equals("concept")) {
						document = addContextDef(document, def).gotoParent();
					}
					token += "::";
				}
			}

			for (org.w3c.dom.Element x : document.gotoRoot().getChildElement()) {
				try {
					res.add(asStringElement(x));
				} catch (TransformerException e) {
					throw new RuntimeException(e);
				}
			}
		}

		return res;
	}

	private XMLTag addContextDef(final XMLTag tag, final ContextDef def) {
		tag.addTag(def.getName()).addAttribute("id", def.getId()).addAttribute("label", def.getLabel());
		if (def.getType() != null && !def.getType().isEmpty()) {
			tag.addAttribute("type", def.getType());
		}
		return tag;
	}

	private String asStringElement(final org.w3c.dom.Element element) throws TransformerException {
		StringWriter buffer = new StringWriter();
		transformer.transform(new DOMSource(element), new StreamResult(buffer));
		return buffer.toString();
	}

	@SuppressWarnings("unchecked")
	private String getRelFundingTree(final String xmlTree) {
		String funding = "<funding>";
		try {
			Document ftree = new SAXReader().read(new StringReader(xmlTree));
			funding = "<funding>";
			// String _id = "";

			for (Object o : Lists.reverse(ftree.selectNodes("//fundingtree//*[starts-with(local-name(),'funding_level_')]"))) {
				Element e = (Element) o;
				String _id = e.valueOf("./id");
				funding += "<" + e.getName() + ">" + escapeXml(_id) + "</" + e.getName() + ">";
				// _id += "::";
			}
		} catch (DocumentException e) {
			System.err.println("unable to parse funding tree: " + xmlTree + "\n" + e.getMessage());
		} finally {
			funding += "</funding>";
		}
		return funding;
	}

	private void fillContextMap(final String xmlTree) {

		Document fundingPath;
		try {
			fundingPath = new SAXReader().read(new StringReader(xmlTree));
		} catch (DocumentException e) {
			throw new RuntimeException(e);
		}
		Node funder = fundingPath.selectSingleNode("//funder");
		String funderShortName = funder.valueOf("./shortname");
		contextMapper.put(funderShortName, new ContextDef(funderShortName, funder.valueOf("./name"), "context", "funding"));
		Node level0 = fundingPath.selectSingleNode("//funding_level_0");
		String level0Id = Joiner.on("::").join(funderShortName, level0.valueOf("./name"));
		contextMapper.put(level0Id, new ContextDef(level0Id, level0.valueOf("./description"), "category", ""));
		Node level1 = fundingPath.selectSingleNode("//funding_level_1");
		String level1Id = Joiner.on("::").join(level0Id, level1.valueOf("./name"));
		contextMapper.put(level1Id, new ContextDef(level1Id, level1.valueOf("./description"), "concept", ""));
		Node level2 = fundingPath.selectSingleNode("//funding_level_2");
		if (level2 == null) {
			contextes.add(level1Id);
		}
		else {
			String level2Id = Joiner.on("::").join(level1Id, level2.valueOf("./name"));
			contextMapper.put(level2Id, new ContextDef(level2Id, level2.valueOf("./description"), "concept", ""));
			contextes.add(level2Id);
		}

	}

	private String asXmlElement(final String name, final String value, final Qualifier q, final DataInfo dataInfo) {
		StringBuilder sb = new StringBuilder();
		sb.append("<");
		sb.append(name);
		if (q != null) {
			sb.append(getAttributes(q));
		}
		if (dataInfo != null) {
			sb = dataInfoAsAttributes(sb, dataInfo);
		}
		if (value == null || value.isEmpty()) {
			sb.append("/>");
			return sb.toString();
			// return "<" + name + getAttributes(q) + "/>";
		}

		sb.append(">");
		// sb.append(escapeXml(Normalizer.normalize(value, Normalizer.Form.NFD)));
		sb.append(escapeXml(value));
		sb.append("</");
		sb.append(name);
		sb.append(">");

		return sb.toString();
		// return "<" + name + getAttributes(q) + ">" + escapeXml(value) + "</" + name + ">";
	}

	private String getAttributes(final Qualifier q) {
		if (q == null) { return ""; }

		StringBuilder sb = new StringBuilder();
		for (Entry<FieldDescriptor, Object> e : q.getAllFields().entrySet()) {
			// sb.append(" " + e.getKey().getName() + "=\"" + escapeXml(e.getValue().toString()) + "\"");
			sb.append(" ");
			sb.append(e.getKey().getName());
			sb.append("=\"");
			sb.append(escapeXml(e.getValue().toString()));
			sb.append("\"");
		}
		return sb.toString();
	}

	private Set<Entry<FieldDescriptor, Object>> filterFields(final GeneratedMessage fields, final Set<String> filter) {

		if (filter != null) {
			Predicate<FieldDescriptor> p = new Predicate<FieldDescriptor>() {

				@Override
				public boolean apply(final FieldDescriptor descriptor) {
					if (fields == null) {
					return false;
					}
					String name = descriptor.getName();
					return filter.contains(name);
				}
			};
			Map<FieldDescriptor, Object> filtered = Maps.filterKeys(fields.getAllFields(), p);
			// System.out.println(
			// "filtered " + type.toString() + ": " + toString(filterEntries.keySet()) + "\n" +
			// "builder  " + fields.getDescriptorForType().getFullName() + ": " + toString(fields.getAllFields().keySet()));
			return filtered.entrySet();
		}
		return fields.getAllFields().entrySet();
	}

	public static String removePrefix(final String s) {
		if (s.contains("|")) { return StringUtils.substringAfter(s, "|"); }
		return s;
	}

	public static String escapeXml(final String value) {
		// return StringEscapeUtils.escapeXml(value).replaceAll("&", "&amp;").replaceAll("<", "&lt;").replaceAll(">", "&gt;");
		return value.replaceAll("&", "&amp;").replaceAll("<", "&lt;").replaceAll(">", "&gt;");
	}

	private List<String> countersAsXml() {
		List<String> out = Lists.newArrayList();
		for (Entry<String, Integer> e : counters.entrySet()) {
			out.add(String.format("<counter_%s value=\"%s\"/>", e.getKey(), e.getValue()));
		}
		return out;
	}

	private void incrementCounter(final String type) {
		if (!counters.containsKey(type)) {
			counters.put(type, 1);
		} else {
			counters.put(type, counters.get(type) + 1);
		}
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("################################################\n");
		sb.append("ID: ").append(key).append("\n");
		if (mainEntity != null) {
			sb.append("MAIN ENTITY:\n").append(mainEntity.getEntity().toString() + "\n");
		}
		if (relations != null) {
			sb.append("\nRELATIONS:\n");
			for (OafDecoder decoder : relations) {
				sb.append(decoder.getOafRel().toString() + "\n");
			}
		}
		if (children != null) {
			sb.append("\nCHILDREN:\n");
			for (OafDecoder decoder : children) {
				sb.append(decoder.getOafRel().toString() + "\n");
			}
		}
		return sb.toString();
	}

}
