package eu.dnetlib.data.mapreduce.hbase.dedup;

import java.io.IOException;

import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableReducer;
import org.apache.hadoop.hbase.util.Bytes;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;

import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.data.mapreduce.util.DedupUtils;
import eu.dnetlib.data.proto.DedupProtos.Dedup;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.TypeProtos.Type;
import eu.dnetlib.data.transform.xml.AbstractDNetOafXsltFunctions;
import eu.dnetlib.pace.config.DedupConfig;
import eu.dnetlib.pace.model.gt.Authors;
import eu.dnetlib.pace.model.gt.CoAuthors;
import eu.dnetlib.pace.model.gt.GTAuthor;
import eu.dnetlib.pace.model.gt.GTAuthorMapper;

public class DedupFindRootsPersonReducer extends TableReducer<ImmutableBytesWritable, ImmutableBytesWritable, ImmutableBytesWritable> {

	private static final boolean WRITE_TO_WAL = false;

	private DedupConfig dedupConf;

	private ImmutableBytesWritable outKey;

	@Override
	protected void setup(final Context context) throws IOException, InterruptedException {
		dedupConf = DedupConfig.load(context.getConfiguration().get(JobParams.DEDUP_CONF));
		System.out.println("dedup findRoots mapper\nwf conf: " + dedupConf.toString());

		outKey = new ImmutableBytesWritable();
	}

	@Override
	protected void reduce(final ImmutableBytesWritable key, final Iterable<ImmutableBytesWritable> values, final Context context) throws IOException,
			InterruptedException {

		final Authors aas = new Authors();
		final CoAuthors cas = new CoAuthors();
		// final Set<String> dupIds = Sets.newHashSet();

		for (final GTAuthor a : asGTA(context, values)) {
			aas.addAll(a.getMerged());
			cas.addAll(a.getCoAuthors());

			// dupIds.add(a.getId());

			final byte[] row = Bytes.toBytes(a.getId());
			final Delete delete = new Delete(row);
			outKey.set(row);
			context.write(outKey, delete);
			context.getCounter(dedupConf.getWf().getEntityType(), "deleted").increment(1);
		}

		final String rootId = hashCodeString(aas);
		final GTAuthor gta = new GTAuthor(rootId, aas, cas, true);

		// for (final String id : dupIds) {
		// final byte[] row = Bytes.toBytes(id);
		// final byte[] root = Bytes.toBytes(rootId);
		// emitDedupRel(context, DedupUtils.getDedupCF_mergedInBytes(Type.person), row, root, buildRel(row, root,
		// Dedup.RelName.isMergedIn));
		// emitDedupRel(context, DedupUtils.getDedupCF_mergesBytes(Type.person), root, row, buildRel(root, row, Dedup.RelName.merges));
		//
		// context.getCounter(dedupConf.getWf().getEntityType(), "dedupRel (x2)").increment(1);
		// }

		final Put put = new Put(Bytes.toBytes(gta.getId()));
		put.setWriteToWAL(WRITE_TO_WAL);
		put.add(Bytes.toBytes(dedupConf.getWf().getEntityType()), DedupUtils.BODY_B, toOafByteArray(gta));

		outKey.set(Bytes.toBytes(gta.getId()));
		context.write(outKey, put);

		context.getCounter(dedupConf.getWf().getEntityType(), "out").increment(1);
	}

	private byte[] buildRel(final byte[] from, final byte[] to, final Dedup.RelName relClass) {
		final OafRel.Builder oafRel = DedupUtils.getDedup(dedupConf, new String(from), new String(to), relClass);
		final Oaf oaf =
				Oaf.newBuilder()
						.setKind(Kind.relation)
						.setTimestamp(System.currentTimeMillis())
						.setDataInfo(
								AbstractDNetOafXsltFunctions.getDataInfo(null, "", "0.8", false, true).setInferenceprovenance(
										dedupConf.getWf().getConfigurationId())).setRel(oafRel)
						.build();
		return oaf.toByteArray();
	}

	private void emitDedupRel(final Context context, final byte[] cf, final byte[] from, final byte[] to, final byte[] value) throws
			IOException, InterruptedException {
		final Put put = new Put(from).add(cf, to, value);
		put.setWriteToWAL(WRITE_TO_WAL);
		context.write(new ImmutableBytesWritable(from), put);
	}

	private Iterable<GTAuthor> asGTA(final Context context, final Iterable<ImmutableBytesWritable> values) {

		return Iterables.transform(values, new Function<ImmutableBytesWritable, GTAuthor>() {

			@Override
			public GTAuthor apply(final ImmutableBytesWritable input) {
				return GTAuthor.fromJson(new String(input.copyBytes()));
			}
		});
	}

	public byte[] toOafByteArray(final GTAuthor gta) {
		final Oaf oaf = new GTAuthorMapper().map(gta);
		return oaf.toByteArray();
	}

	protected String hashCodeString(final Authors ag) {
		return getRowKey(String.valueOf(ag.hashCode()));
	}

	protected String getRowKey(final String s) {
		return AbstractDNetOafXsltFunctions.oafId(Type.person.toString(), "dedup_wf_001", s);
	}

}
