package eu.dnetlib.data.mapreduce.hbase.dataimport;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Mapper;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import eu.dnetlib.data.mapreduce.JobParams;
import eu.dnetlib.data.mapreduce.util.DedupUtils;
import eu.dnetlib.data.mapreduce.util.OafDecoder;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.Oaf.Builder;
import eu.dnetlib.data.proto.PersonProtos.Person.CoAuthor;
import eu.dnetlib.data.proto.TypeProtos;
import eu.dnetlib.data.transform.Column;
import eu.dnetlib.data.transform.Row;
import eu.dnetlib.data.transform.XsltRowTransformer;
import eu.dnetlib.data.transform.XsltRowTransformerFactory;

public class AuthorImportRecordsMapper extends Mapper<Text, Text, ImmutableBytesWritable, Put> {

	/**
	 * logger.
	 */
	private static final Log log = LogFactory.getLog(AuthorImportRecordsMapper.class); // NOPMD by marko on 11/24/08 5:02 PM

	private static final boolean WRITE_TO_WAL = false;

	private XsltRowTransformer transformer;

	private ImmutableBytesWritable ibw;

	private Map<String, String> merged;

	@Override
	protected void setup(final Context context) throws IOException, InterruptedException {
		super.setup(context);

		final String xslt = context.getConfiguration().get(JobParams.HBASE_IMPORT_XSLT).trim();

		if ((xslt == null) || xslt.isEmpty()) throw new IllegalArgumentException("missing xslt");
		final Map<String, Object> xslParams = Maps.newHashMap();

		if (context.getConfiguration().get("datasourceTypeMap") != null) {
			final Map<String, String> dsMap = getDatasourceTypeMap(context);

			log.info("using datasource type map:\n" + dsMap.toString());

			xslParams.put("mergeIdForHomonymsMap", dsMap);
		}

		transformer = XsltRowTransformerFactory.newInstance(xslt, xslParams);

		ibw = new ImmutableBytesWritable();

		merged = loadMap(context.getConfiguration());

		log.info("map { native id --> anchor id } size: " + merged.size());
		log.info("got xslt: '" + xslt);
		log.info("using trasformer: '" + transformer.getTransformerClassName() + "'");
	}

	@Override
	protected void map(final Text key, final Text value, final Context context) throws IOException, InterruptedException {
		try {
			final List<Row> rowList = transformer.apply(value.toString());

			context.getCounter("mdstore", "person (from mdstore)").increment(rowList.size());

			final Iterable<Builder> oafBuilders = Iterables.transform(rowList, new Function<Row, Oaf.Builder>() {

				@Override
				public Builder apply(final Row row) {
					final Column<String, byte[]> col = row.getColumn(DedupUtils.BODY_S);
					final OafDecoder d = OafDecoder.decode(col.getValue());
					return Oaf.newBuilder(d.getOaf());
				}
			});

			final Iterable<Oaf.Builder> filtered = Iterables.filter(oafBuilders, new Predicate<Oaf.Builder>() {

				@Override
				public boolean apply(final Oaf.Builder b) {
					final boolean res = merged.containsKey(b.getEntity().getId());
					if (res) {
						context.getCounter("mdstore", "person (found in anchor map)").increment(1);
					}
					return !res;
				}
			});

			final Iterable<Row> patched = Iterables.transform(filtered, new Function<Oaf.Builder, Row>() {

				@Override
				public Row apply(final Oaf.Builder b) {

					for (final CoAuthor.Builder cb : b.getEntityBuilder().getPersonBuilder().getCoauthorBuilderList()) {
						final String anchorId = merged.get(cb.getId());
						if (StringUtils.isNotBlank(anchorId)) {
							context.getCounter("mdstore", "person (coAuthor as anchor)").increment(1);
							cb.setAnchorId(anchorId);
						}
					}

					final String family = TypeProtos.Type.person.toString();
					final Row rowout = new Row(family, b.getEntity().getId());
					rowout.setColumn(family, Column.newInstance(DedupUtils.BODY_S, b.build().toByteArray()));

					return rowout;
				}
			});

			write(context, patched);
		} catch (final Throwable e) {
			log.error("error importing the following record on HBase: " + value.toString(), e);
			context.getCounter("error", e.getClass().getName()).increment(1);
			throw new RuntimeException(e);
		}
	}

	private void write(final Context context, final Iterable<Row> filtered) throws IOException, InterruptedException {
		for (final Row row : filtered) {

			final byte[] rowKey = Bytes.toBytes(row.getKey());
			final Put put = new Put(rowKey);
			put.setWriteToWAL(WRITE_TO_WAL);

			for (final Column<String, byte[]> col : row) {
				final byte[] family = Bytes.toBytes(row.getColumnFamily());
				final byte[] qualifier = Bytes.toBytes(col.getName());
				put.add(family, qualifier, col.getValue());
			}
			ibw.set(rowKey);
			context.write(ibw, put);
			context.getCounter("mdstore", row.getColumnFamily()).increment(row.getColumns().size());
		}
	}

	private Map<String, String> getDatasourceTypeMap(final Context context) {
		final String dsType = context.getConfiguration().get("datasourceTypeMap");

		final Type token = new TypeToken<Map<String, String>>() {}.getType();
		final Map<String, String> dsMap = new Gson().fromJson(dsType, token);
		return dsMap;
	}

	private Map<String, String> loadMap(final Configuration conf) throws IOException {
		final Map<String, String> map = Maps.newHashMap();
		final String filePath = conf.get("mapred.output.dir") + "/part-r-00000";
		if (StringUtils.isBlank(filePath)) throw new IllegalArgumentException("missing 'mapred.output.dir'");

		final Path path = new Path(filePath);
		final BufferedReader br = new BufferedReader(new InputStreamReader(FileSystem.get(conf).open(path)));

		String line = br.readLine();
		while (line != null) {
			final String[] split = line.split("=");

			map.put(split[0], split[1]);

			line = br.readLine();
		}

		return map;
	}

}
