package eu.dnetlib.data.collector.plugins.oai;

import java.util.Iterator;
import java.util.List;

import org.springframework.beans.factory.annotation.Required;

import com.google.common.base.Function;
import com.google.common.base.Splitter;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;

import eu.dnetlib.data.collector.plugin.CollectorPlugin;
import eu.dnetlib.data.collector.rmi.CollectorServiceException;
import eu.dnetlib.data.collector.rmi.InterfaceDescriptor;

public class OaiCollectorPlugin implements CollectorPlugin {
	
	private static final String FORMAT_PARAM = "format";
	private static final String OAI_SET_PARAM = "set";
	
	private OaiIteratorFactory oaiIteratorFactory;

	@Override
	public Iterable<String> collect(final InterfaceDescriptor interfaceDescriptor, final String fromDate, final String untilDate) throws CollectorServiceException {
		final String baseUrl = interfaceDescriptor.getBaseUrl();
		final String mdFormat = interfaceDescriptor.getParams().get(FORMAT_PARAM);
		final String setParam = interfaceDescriptor.getParams().get(OAI_SET_PARAM);
		final List<String> sets = Lists.newArrayList();
		if (setParam != null) {
			sets.addAll(Lists.newArrayList(Splitter.on(",").omitEmptyStrings().trimResults().split(setParam)));
		}
		if (sets.isEmpty()) {
			// If no set is defined, ALL the sets must be harvested
			sets.add("");
		}
		
		if ((baseUrl == null) || baseUrl.isEmpty()) {
			throw new CollectorServiceException("Param 'baseurl' is null or empty");
		}

		if ((mdFormat == null) || mdFormat.isEmpty()) {
			throw new CollectorServiceException("Param 'mdFormat' is null or empty");
		}

		if ((fromDate != null) && !fromDate.matches("\\d{4}-\\d{2}-\\d{2}")) {
			throw new CollectorServiceException("Invalid date (YYYY-MM-DD): " + fromDate);
		}
		
		if ((untilDate != null) && !untilDate.matches("\\d{4}-\\d{2}-\\d{2}")) {
			throw new CollectorServiceException("Invalid date (YYYY-MM-DD): " + untilDate);
		}

		return new Iterable<String>() {
			@SuppressWarnings("unchecked")
			@Override
			public Iterator<String> iterator() {
				final Iterable<Iterator<String>> iter = Iterables.transform(sets, new Function<String, Iterator<String>>() {
					@Override
					public Iterator<String> apply(String set) {
						return oaiIteratorFactory.newIterator(baseUrl, mdFormat, set, fromDate, untilDate);
					}
				});
				return Iterators.concat(Iterables.toArray(iter, Iterator.class));
			}
		};
	}

	@Override
	public String getProtocol() {
		return "OAI";
	}

	public OaiIteratorFactory getOaiIteratorFactory() {
		return oaiIteratorFactory;
	}

	@Required
	public void setOaiIteratorFactory(OaiIteratorFactory oaiIteratorFactory) {
		this.oaiIteratorFactory = oaiIteratorFactory;
	}

	@Override
	public List<String> listNameParameters() {
		return Lists.newArrayList(OAI_SET_PARAM, FORMAT_PARAM);
	}

}
