package eu.dnetlib.data.collector.plugins.ftp;

import java.io.IOException;
import java.util.Collection;
import java.util.concurrent.BlockingQueue;

import org.apache.commons.io.filefilter.FileFilterUtils;
import org.apache.commons.io.filefilter.HiddenFileFilter;
import org.apache.commons.io.filefilter.IOFileFilter;
import org.apache.commons.io.filefilter.SuffixFileFilter;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.net.ftp.FTPFile;
import org.apache.commons.net.ftp.FTPFileFilter;
import org.xml.sax.SAXException;

/**
 * FTPFileWalker runs recursively under a FTP directory structure starting from a given path and for each file reads it's content and puts
 * it in a shared queue.
 * 
 * Acts as a producer.
 * 
 * @author sandro
 * 
 * @param <T>
 *            Type of expected content to extract from files NB. Generally strings.
 */
public class FtpFileWalker<T> extends FtpDirectoryWalker<T> {

	/**
	 * Logger.
	 */
	private static final Log log = LogFactory.getLog(FtpFileWalker.class);

	/**
	 * Shared queue.
	 */
	private final BlockingQueue<T> queue;

	public static String IGNORE_PREFIX = ".";

	public static String IGNORE_SUFFIX = "~";

	static IOFileFilter fileFilter = FileFilterUtils.notFileFilter(FileFilterUtils.or(new SuffixFileFilter("~"), HiddenFileFilter.HIDDEN));

	/**
	 * Instantiates a new fTP file walker.
	 * 
	 * @param queue
	 *            the queue
	 * @param filter
	 *            the filter
	 * @param depthLimit
	 *            the depth limit
	 * @param itemParam
	 *            the item param
	 * @param protocol
	 *            the protocol
	 */
	public FtpFileWalker(final BlockingQueue<T> queue, final FTPFileFilter filter, final int depthLimit, final ItemUtility itemParam, final String protocol) {
		super(filter, depthLimit, itemParam, protocol);
		this.queue = queue;
	}

	@SuppressWarnings("unchecked")
	public void doWalk(final String source) throws IOException {
		try {
			initialize();
		} catch (Exception e) {
			enqueue(queue, (T) FtpIterable.done);
			throw new IllegalStateException(e);
		}

		FTPFile[] files = clientProvider.listFiles(source);
		if (files.length == 1) {
			if (source.contains(files[0].getName())) {
				enqueue(queue, (T) getFileFrom(source));
			} else {
				walk(source, queue);
			}
		} else if (files.length > 0) {
			walk(source, queue);
		}
		enqueue(queue, (T) FtpIterable.done);
	}

	@Override
	@SuppressWarnings({ "unchecked", "rawtypes" })
	protected void handleFile(final FTPFile file, final int depth, final Collection results) throws IOException {
		if (file.getName().endsWith(IGNORE_SUFFIX)) return;
		T myFile = (T) readFile(file);
		if (myFile == null) return;
		enqueue((BlockingQueue<T>) results, myFile);
	}

	@Override
	@SuppressWarnings({ "unchecked", "rawtypes" })
	protected boolean handleDirectory(final String directory, final int depth, final Collection results) throws IOException {
		String[] tmp = directory.split("/");
		if ((tmp.length > 0) && (tmp[tmp.length - 1].startsWith(IGNORE_PREFIX))) return false;
		return super.handleDirectory(directory, depth, results);
	}

	// ///Utility

	/**
	 * Adds the element to the queue
	 */
	private void enqueue(final BlockingQueue<T> queue, final T element) {
		try {
			queue.put(element);
			// In order to avoid orphan FTP walkers, a better version would be..
			// if (!queue.offer(element, 5, TimeUnit.MINUTES)) {
			// log.info("Nobody is consuming my records! I stop gracefully!");
			// }
		} catch (InterruptedException e) {
			log.warn("Hopssss... ", e);
		}
	}

	/**
	 * given a file, return its content as a string
	 * 
	 * @param file
	 *            the source
	 * @return the file content as a single string
	 * @throws IOException
	 * @throws TikaException
	 * @throws SAXException
	 */
	private String readFile(final FTPFile file) throws IOException {

		if (getCurrentDirectory() == null) {
			clientProvider.completePendingCommand();
			return getCurrentDirectory();
		}
		String fileContent = getFileFrom(getCurrentDirectory() + "/" + file.getName());
		// remove BOM if present
		if (fileContent.startsWith("\uFEFF")) {
			fileContent = fileContent.substring(1);
		}
		return fileContent;
	}

}
