package eu.dnetlib.data.collector.plugins.oai.engine;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.GeneralSecurityException;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.List;
import java.util.Map;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import eu.dnetlib.data.collector.rmi.CollectorServiceException;

/**
 * @author jochen, michele
 *
 */
public class HttpConnector {

	private static final Log log = LogFactory.getLog(HttpConnector.class);

	private int maxNumberOfRetry = 6; 
	private int defaultDelay = 120; //seconds
	private int readTimeOut = 120; //seconds

	/**
	 * @param requestUrl
	 * @return the content of the downloaded resource 
	 * @throws CollectorServiceException
	 */
	public String getInputSource(final String requestUrl) throws CollectorServiceException {
		return attemptDownload(requestUrl, 1);
	}

	private String attemptDownload(final String requestUrl, final int retryNumber) throws CollectorServiceException {
		
		if (retryNumber > maxNumberOfRetry) {
			throw new CollectorServiceException("Max number of retries exceeded");
		}

		log.debug("Downloading " + requestUrl + " - try: " + retryNumber);
		try {
			InputStream input = null;
			
			try {
				final HttpURLConnection urlConn = (HttpURLConnection) (new URL(requestUrl)).openConnection();
				urlConn.setReadTimeout(readTimeOut * 1000);

				if (log.isDebugEnabled()) {
					logHeaderFields(urlConn);
				}

				int retryAfter = obtainRetryAfter(urlConn.getHeaderFields());
				if (retryAfter > 0) {
					log.warn("waiting and repeating request after " + retryAfter + " sec.");
					Thread.sleep(retryAfter * 1000);
					return attemptDownload(requestUrl, retryNumber + 1);
				} else if (urlConn.getResponseCode() == HttpURLConnection.HTTP_MOVED_PERM || urlConn.getResponseCode() == HttpURLConnection.HTTP_MOVED_TEMP){
					final String newUrl = obtainNewLocation(urlConn.getHeaderFields());
					log.info("The requested url has been moved to " + newUrl);
					return attemptDownload(newUrl, retryNumber + 1);
				} else if ((urlConn.getResponseCode() != HttpURLConnection.HTTP_OK)) {
					log.error("HTTP error: " + urlConn.getResponseMessage());
					Thread.sleep(defaultDelay * 1000);
					return attemptDownload(requestUrl, retryNumber + 1);
				} else {
					input = urlConn.getInputStream();
					return IOUtils.toString(input);
				}
			} catch (IOException e) {
				log.error("error while retrieving from http-connection occured: " + e);
				Thread.sleep(defaultDelay * 1000);
				return attemptDownload(requestUrl, retryNumber + 1);
			} finally {
				IOUtils.closeQuietly(input);
			}
		} catch (InterruptedException e) {
			throw new CollectorServiceException(e);
		}
	}

	private void logHeaderFields(final HttpURLConnection urlConn) throws IOException {
		log.debug("StatusCode: " + urlConn.getResponseMessage());

		for (Map.Entry<String, List<String>> e : urlConn.getHeaderFields().entrySet()){
			if (e.getKey() != null) {
				for (String v : e.getValue()) {
					log.debug("  key: " + e.getKey() + " - value: " + v);
				}
			}
		}
	}

	private int obtainRetryAfter(final Map<String, List<String>> headerMap) {
		for (String key : headerMap.keySet()){
			if (key != null	&& key.toLowerCase().equals("retry-after") && headerMap.get(key).size() > 0 && NumberUtils.isNumber(headerMap.get(key).get(0))) {
				return Integer.parseInt(headerMap.get(key).get(0)) + 10;
			}
		}
		return -1;
	}

	private String obtainNewLocation(final Map<String, List<String>> headerMap) throws CollectorServiceException {
		for (String key : headerMap.keySet()){
			if (key != null && key.toLowerCase().equals("location") && headerMap.get(key).size() > 0) {
				return headerMap.get(key).get(0);
			}
		}
		throw new CollectorServiceException("The requested url has been MOVED, but 'location' param is MISSING");
	}

	/**
	 * register for https scheme; this is a workaround and not intended for the use in trusted environments
	 * @throws NoSuchAlgorithmException
	 * @throws KeyManagementException
	 */
	public void initTrustManager() {
		final X509TrustManager tm = new X509TrustManager() {
			public void checkClientTrusted(X509Certificate[] xcs, String string) throws CertificateException {}
			public void checkServerTrusted(X509Certificate[] xcs, String string) throws CertificateException {}
			public X509Certificate[] getAcceptedIssuers(){
				return null;
			}
		};
		try {
			final SSLContext ctx = SSLContext.getInstance("TLS");
			ctx.init(null, new TrustManager[]{tm}, null);
			HttpsURLConnection.setDefaultSSLSocketFactory(ctx.getSocketFactory());
		} catch (GeneralSecurityException e) {
			log.fatal(e);
			throw new IllegalStateException(e);
		}
	}

	public int getMaxNumberOfRetry() {
		return maxNumberOfRetry;
	}

	public void setMaxNumberOfRetry(int maxNumberOfRetry) {
		this.maxNumberOfRetry = maxNumberOfRetry;
	}

	public int getDefaultDelay() {
		return defaultDelay;
	}

	public void setDefaultDelay(int defaultDelay) {
		this.defaultDelay = defaultDelay;
	}

	public int getReadTimeOut() {
		return readTimeOut;
	}

	public void setReadTimeOut(int readTimeOut) {
		this.readTimeOut = readTimeOut;
	}
	

}