package eu.dnetlib.data.collector.plugins.ftp;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPFile;
import org.apache.commons.net.ftp.FTPReply;

/**
 * The Class ClientFTPDataProvider.
 */
public class ClientFtpDataProvider implements FtpClientProvider {

	/** The logger. */
	private static final Log log = LogFactory.getLog(ClientFtpDataProvider.class);

	/** The client. */
	private FTPClient client = new FTPClient();

	/** The item param. */
	private ItemUtility itemParam;

	/*
	 * (non-Javadoc)
	 * 
	 * @see eu.dnetlib.data.collective.harvest.provider.IClientProvider#connect()
	 */
	@Override
	public void connect() {
		synchronized (this) {
			try {

				client.connect(itemParam.getHost());
				client.enterLocalPassiveMode();

				if (!checkPositiveResponse()) throw new IllegalStateException("unable to connect to ftp server. " + itemParam.getHost());

				if (!client.login(itemParam.getUsername(), itemParam.getPassword()))
					throw new IllegalStateException("unable to login to ftp server. " + itemParam.getHost());

			} catch (Exception e) {
				throw new IllegalStateException("unable to connect to ftp server. " + itemParam.getHost(), e);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see eu.dnetlib.data.collective.harvest.provider.IClientProvider#disconnect()
	 */
	@Override
	public void disconnect() {
		synchronized (this) {
			try {
				client.disconnect();
			} catch (IOException e) {
				throw new IllegalStateException(e);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see eu.dnetlib.data.collective.harvest.provider.ftp.FTPClientProvider#changeWorkingDirectory(java.lang.String)
	 */
	@Override
	public boolean changeWorkingDirectory(final String path) {
		synchronized (this) {
			Boolean b = false;
			try {
				b = client.changeWorkingDirectory(path);
			} catch (IOException e) {
				return false;
			}
			return b;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see eu.dnetlib.data.collective.harvest.provider.ftp.FTPClientProvider#listFiles()
	 */
	@Override
	public FTPFile[] listFiles() {
		synchronized (this) {
			try {
				return client.listFiles();
			} catch (IOException e) {
				log.error("cannot list files", e);
				throw new IllegalStateException(e);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see eu.dnetlib.data.collective.harvest.provider.ftp.FTPClientProvider#listFiles(java.lang.String)
	 */
	@Override
	public FTPFile[] listFiles(final String source) {
		synchronized (this) {
			try {
				return client.listFiles(source);
			} catch (IOException e) {
				log.error("cannot list files", e);
				throw new IllegalStateException(e);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see eu.dnetlib.data.collective.harvest.provider.ftp.FTPClientProvider#printWorkingDirectory()
	 */
	@Override
	public String printWorkingDirectory() {
		synchronized (this) {
			try {
				return client.printWorkingDirectory();
			} catch (IOException e) {
				log.error("cannot print working directory", e);
				throw new IllegalStateException(e);
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see eu.dnetlib.data.collective.harvest.provider.IClientProvider#retrieveFileStream(java.lang.String)
	 */
	@Override
	public String retrieveFileStream(final String remote) throws IOException {
		int nRepeat = 4;
		while (nRepeat > 0) {
			try {
				return doRetrieveStream(remote);
			} catch (IOException e) {
				nRepeat--;
				log.info("An error occurred, retrying.. ({n} retries left)".replace("{n}", String.valueOf(nRepeat)));
				try {
					Thread.sleep(5000);
				} catch (InterruptedException e1) {
					// TODO Auto-generated catch block
					e1.printStackTrace();
				}
			}
		}
		return "<INVALID src='" + remote + "'/>";
	}

	/**
	 * Do retrieve stream.
	 * 
	 * @param remote
	 *            the remote
	 * @return the string
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	private String doRetrieveStream(final String remote) throws IOException {

		final InputStream stream = client.retrieveFileStream(remote);
		final StringWriter writer = new StringWriter();
		try {
			if (!checkPositiveResponse()) throw new IllegalStateException("unable to retrieve stream from ftp server. " + itemParam.getHost());

			IOUtils.write(IOUtils.toByteArray(stream), writer);

			return writer.toString();
		} catch (Exception e) {
			log.error("Exception retrieving stream from " + remote, e);
			return null;
		}

		finally {
			IOUtils.closeQuietly(writer);
			IOUtils.closeQuietly(stream);
			if (!client.completePendingCommand()) throw new IllegalStateException("unable to complete pending command on ftp server. " + itemParam.getHost());

		}
	}

	/**
	 * Check positive response.
	 * 
	 * @return true, if successful
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	private boolean checkPositiveResponse() throws IOException {
		int reply = client.getReplyCode();
		if (!FTPReply.isPositiveCompletion(reply) & !FTPReply.isPositivePreliminary(reply) & !FTPReply.isPositiveIntermediate(reply)) {
			client.disconnect();
			log.error("FTP server refused connection. " + itemParam.getHost());
			return false;
		}
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see eu.dnetlib.data.collective.harvest.provider.ftp.FTPClientProvider#completePendingCommand()
	 */
	@Override
	public void completePendingCommand() throws IOException {
		synchronized (this) {
			client.completePendingCommand();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * eu.dnetlib.data.collective.harvest.provider.IClientProvider#setItemParam(eu.dnetlib.data.collective.harvest.provider.ItemUtility)
	 */
	@Override
	public void setItemParam(final ItemUtility itemUtility) {
		this.itemParam = itemUtility;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see eu.dnetlib.data.collective.harvest.provider.IClientProvider#isConnected()
	 */
	@Override
	public boolean isConnected() {
		return client.isConnected();
	}
}
