package eu.dnetlib.data.collector.plugins.ftp2;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.SocketException;
import java.net.URL;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Queue;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPFile;
import org.apache.commons.net.ftp.FTPReply;

public class FtpIterator implements Iterator<String> {

	private static final Log log = LogFactory.getLog(FtpIterator.class);

	private static final int MAX_RETRIES = 5;
	private static final int DEFAULT_TIMEOUT = 30000;
	private static final long BACKOFF_MILLIS = 10000;
	
	private FTPClient ftpClient;
	private String ftpServer;
	private String remoteBasePath;
	private int port;
	private String username;
	private String password;
	private boolean isRecursive;
    
	private Queue<String> queue;
	
	public FtpIterator(final String baseUrl, final int port, final  String username, final String password, final boolean isRecursive) {
		this.port = port;
		this.username = username;
		this.password= password;
		this.isRecursive = isRecursive;
		try {
			URL server = new URL(baseUrl);
			this.ftpServer = server.getHost();
			this.remoteBasePath = server.getPath();
		} catch (MalformedURLException e1) {
			throw new RuntimeException("Malformed URL exception" + baseUrl);
		}
		
		try {
			initializeQueue();
		} catch (Exception e) {
			log.error("Error in initializing FTP iterator: " + e.getMessage(), e);
			throw new IllegalStateException(e.getMessage(), e);
		}
	}
	
	private void initializeQueue() throws SocketException, IOException {
		ftpClient = new FTPClient();
		ftpClient.setDefaultTimeout(DEFAULT_TIMEOUT);
		queue = new LinkedList<String>();
		
		ftpClient.setDataTimeout(DEFAULT_TIMEOUT);
		ftpClient.setConnectTimeout(DEFAULT_TIMEOUT);
		ftpClient.connect(ftpServer, port);
		
		if (!checkPositiveResponse()) {
			ftpClient.disconnect();
			throw new IllegalStateException("Unable to connect to ftp server. " + ftpServer);
		}
		
		if(!ftpClient.login(username, password)) {
			ftpClient.logout();
			throw new IllegalStateException("Unable to login to ftp server. " + ftpServer);
		}
		
		ftpClient.enterLocalPassiveMode();
		
		listDirectoryRecursive(ftpClient, remoteBasePath, "");
	}
	
	private void listDirectoryRecursive(FTPClient ftpClient, String parentDir, String currentDir) {
	    String dirToList = parentDir;
	    if (!currentDir.equals("")) {
	        dirToList += "/" + currentDir;
	    }
	    FTPFile[] subFiles;
		try {
			subFiles = ftpClient.listFiles(dirToList);
			if (subFiles != null && subFiles.length > 0) {
				for (FTPFile aFile : subFiles) {
					String currentFileName = aFile.getName();
					if (currentFileName.equals(".") || currentFileName.equals("..")) {
						// skip parent directory and directory itself
						continue;
					}
					if (aFile.isDirectory()) {
						if (isRecursive) {
							listDirectoryRecursive(ftpClient, dirToList, currentFileName);
						}
					} else {
						queue.add(dirToList + "/" + currentFileName);
					}
				}
			}
		} catch (IOException e) {
			throw new IllegalStateException("unable to list FTP folder tree", e);
		}
	}

	@Override
	public boolean hasNext() {
		if (queue.isEmpty()) {
			try {
				ftpClient.logout();
				ftpClient.disconnect();
				return false ;
			} catch (IOException e) {
				throw new IllegalStateException("Failed to logout & close FTP connection", e);
			}
		} else {
			return true;
		}
	}

	@Override
	public String next() {
		final String path = queue.remove();
		int nRepeat = 0;
		while (nRepeat < MAX_RETRIES) {
			InputStream retrievedFileStream = null;
			try {
				retrievedFileStream = ftpClient.retrieveFileStream(path);
				if (!checkPositiveResponse()) 
					throw new IllegalStateException("Unable to retrieve stream from ftp server. " + ftpServer + path);
				
				if (retrievedFileStream != null) {
					log.info("Collected file from FTP: " + ftpServer + path);
					return IOUtils.toString(retrievedFileStream);
				} else {
					throw new IllegalStateException("File retrieved from " + path + " is NULL");
				}
			} catch (IOException e) {
				nRepeat++;
				log.info("An error occurred for " + ftpServer + path + ", retrying.. [retried {n} time(s)]".replace("{n}", String.valueOf(nRepeat)));
				try {
					Thread.sleep(BACKOFF_MILLIS);
				} catch (InterruptedException e1) {
					// TODO Auto-generated catch block
					e1.printStackTrace();
				}
			} finally {
				try {
					if (retrievedFileStream != null) {
						IOUtils.closeQuietly(retrievedFileStream);
					}
					if (!ftpClient.completePendingCommand())
						throw new IllegalStateException("unable to complete pending command on ftp server. " + ftpServer);
				} catch (IOException e) {
					log.error("Cannot execute ftpClient.completePendingCommand");
				}
			}
		}
		throw new IllegalStateException("Impossible to retrieve FTP file " + path + " after " + nRepeat + " retries. Aborting FTP collection.");
	}

	@Override
	public void remove() {
		// TODO Auto-generated method stub
	}
	
	/**
	 * Check positive response.
	 * 
	 * @return true, if successful
	 * @throws IOException
	 *             Signals that an I/O exception has occurred.
	 */
	private boolean checkPositiveResponse() throws IOException {
		int reply = ftpClient.getReplyCode();
		if (!FTPReply.isPositiveCompletion(reply) & !FTPReply.isPositivePreliminary(reply) & !FTPReply.isPositiveIntermediate(reply)) {
			ftpClient.disconnect();
			log.error("FTP server refused connection. " + ftpServer);
			throw new RuntimeException("FTP server refused connection. " + ftpServer);
		}
		return true;
	}
}
