package eu.dnetlib.functionality.modular.ui.lightui.clients;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;

import com.google.common.collect.Lists;

import eu.dnetlib.data.provision.index.rmi.IndexServiceException;
import eu.dnetlib.enabling.resultset.rmi.ResultSetException;
import eu.dnetlib.functionality.index.client.IndexClient;
import eu.dnetlib.functionality.index.client.IndexClientException;
import eu.dnetlib.functionality.index.client.ResolvingIndexClientFactory;
import eu.dnetlib.functionality.index.client.response.BrowseEntry;
import eu.dnetlib.functionality.index.client.response.BrowseValueEntry;
import eu.dnetlib.functionality.index.client.response.LookupResponse;
import eu.dnetlib.functionality.modular.ui.lightui.objects.IndexConfiguration;
import eu.dnetlib.functionality.modular.ui.lightui.objects.SearchResult;
import eu.dnetlib.miscutils.functional.xml.ApplyXslt;

public class IndexLightUIClient {

	@Autowired
	private ResolvingIndexClientFactory indexClientFactory;

	private Map<String, IndexClient> indexClientMap;

	private static final Log log = LogFactory.getLog(IndexLightUIClient.class);

	/**
	 * Fixes the labels of the index response according to the specification of the given map.
	 * <p>
	 * If a browse entry has label = x and field=xForBrowsing, and the map contains (x, xForUI), the returned BrowseEntry will contain label
	 * = xForUI, field=xForBrowsing.
	 * </p>
	 * 
	 * @param browseResult
	 *            List of BrowseEntry
	 * @param fieldLabels
	 * @return
	 */
	private List<BrowseEntry> fixBrowseEntryLabels(final List<BrowseEntry> browseResult, final Map<String, String> fieldLabels) {
		List<BrowseEntry> fixedBrowseLabels = Lists.newArrayListWithExpectedSize(browseResult.size());
		for (BrowseEntry be : browseResult) {
			String fieldName = be.getLabel();
			String browsableFieldName = be.getField();
			if (fieldLabels.containsKey(fieldName)) {
				fixedBrowseLabels.add(new BrowseEntry(browsableFieldName, fieldLabels.get(fieldName), be.getValues()));
			} else {
				fixedBrowseLabels.add(be);
			}
		}
		return fixedBrowseLabels;
	}

	/**
	 * 
	 * @param query
	 * @param idx
	 * @param browseFieldLabels
	 *            map with key = field name to browse for, value = label to use in the UI
	 * @param max
	 * @return
	 * @throws IndexServiceException
	 * @throws DocumentException
	 * @throws ResultSetException
	 */
	public List<BrowseEntry> browse(final String query, final IndexConfiguration idx, final Map<String, String> browseFieldLabels, final int max)
			throws IndexServiceException, DocumentException, ResultSetException {

		final List<String> browseFields = new ArrayList<String>(browseFieldLabels.keySet());

		IndexClient indexClient = getIndexClient(idx);
		List<BrowseEntry> browseResult = indexClient.browse(query, browseFields, max);

		return fixBrowseEntryLabels(browseResult, browseFieldLabels);

	}

	/**
	 * 
	 * @param query
	 * @param idx
	 * @param browseFieldLabels
	 *            map with key = field name to browse for, value = label to use in the UI
	 * @param max
	 * @param xslt
	 * @return
	 * @throws IndexServiceException
	 * @throws DocumentException
	 * @throws ResultSetException
	 */
	public String browse(final String query, final IndexConfiguration idx, final Map<String, String> browseFieldLabels, final int max, final Resource xslt)
			throws IndexServiceException, DocumentException, ResultSetException {
		List<BrowseEntry> myResults = browse(query, idx, browseFieldLabels, max);
		int total = myResults.size();

		if (total > 0) {

			final Element docRoot = DocumentHelper.createElement("browse");
			final Document docRes = DocumentHelper.createDocument(docRoot);

			for (BrowseEntry myEntry : myResults) {

				String xpath = "./field[@name='" + myEntry.getField() + "']";

				Element f = (Element) docRoot.selectSingleNode(xpath);
				if (f == null) {
					f = docRoot.addElement("field");
					f.addAttribute("name", myEntry.getField());
					f.addAttribute("label", myEntry.getLabel());
				}

				for (BrowseValueEntry valueEntry : myEntry.getValues()) {
					final Element v = f.addElement("value");
					v.addAttribute("name", valueEntry.getValue().replaceAll("\"", "%22").replaceAll("'", "%27"));
					v.addAttribute("size", "" + valueEntry.getSize());
				}

			}

			return new ApplyXslt(xslt).evaluate(docRes.asXML());
		}

		return "";
	}

	public SearchResult search(final String query, final IndexConfiguration idx, final int page, final int pageSize, final Resource xslt)
			throws IndexServiceException, DocumentException, ResultSetException {

		log.info("executing query: " + query);
		IndexClient indexClient = getIndexClient(idx);
		int prod = page * pageSize;
		final int from = prod - pageSize;
		final int to = (from + pageSize) - 1;
		LookupResponse lookupResponse = indexClient.lookup(query, null, from, to);

		final long total = lookupResponse.getTotal();
		final StringWriter html = new StringWriter();

		ApplyXslt f = new ApplyXslt(xslt);
		int totalPages = 0;
		if (lookupResponse.getRecords() != null) {
			for (String s : lookupResponse.getRecords()) {
				html.append(f.evaluate(s));
			}
			totalPages = ((int) lookupResponse.getTotal() / pageSize) + 1;
		}

		return new SearchResult(page, totalPages, total, html.toString());
	}

	public String getDocument(final IndexConfiguration idx, final String field, final String id, final Resource xslt) throws IndexServiceException,
			ResultSetException {

		final String query = field + " exact \"" + id.trim() + "\"";
		IndexClient indexClient = getIndexClient(idx);
		LookupResponse lookupResponse = indexClient.lookup(query, null, 0, 1);

		final int total = (int) lookupResponse.getTotal();
		if (total != 1) {
			log.error("Invalid number of results (" + total + ") for query: " + query);
			throw new IllegalArgumentException("Invalid number of results (" + total + ") for query: " + query);
		}

		final String record = lookupResponse.getRecords().get(0);

		return new ApplyXslt(xslt).evaluate(record);
	}

	private IndexClient getIndexClient(final IndexConfiguration configuration) throws IndexClientException {
		if (indexClientMap == null) {
			indexClientMap = new HashMap<String, IndexClient>();
		}
		if (indexClientMap.containsKey(configuration.getBackendId())) return indexClientMap.get(configuration.getBackendId());
		IndexClient index = indexClientFactory.getClient(configuration.getFormat(), configuration.getLayout(), configuration.getInterpretation(),
				configuration.getBackendId());
		indexClientMap.put(configuration.getBackendId(), index);

		return index;
	}
}
