package eu.dnetlib.enabling.datasources;

import java.io.StringReader;
import java.text.ParseException;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import javax.xml.ws.wsaddressing.W3CEndpointReference;

import org.antlr.stringtemplate.StringTemplate;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Document;
import org.dom4j.io.SAXReader;
import org.quartz.CronExpression;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.Resource;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

import eu.dnetlib.enabling.database.rmi.DatabaseException;
import eu.dnetlib.enabling.database.rmi.DatabaseService;
import eu.dnetlib.enabling.datasources.rmi.DatasourceDesc;
import eu.dnetlib.enabling.datasources.rmi.DatasourceManagerServiceException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpDocumentNotFoundException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.is.registry.rmi.ISRegistryService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;
import eu.dnetlib.miscutils.functional.UnaryFunction;
import eu.dnetlib.miscutils.functional.xml.ApplyXslt;

public class DatasourceManagerClients {

	private static final Resource xslt = new ClassPathResource("/eu/dnetlib/enabling/datasources/repo_2_is.xslt");
	private static final String REPOSITORY_SERVICE_RESOURCE_TYPE = "RepositoryServiceResourceType";
	private static final Log log = LogFactory.getLog(DatasourceManagerClients.class);
	private static final String TMPLS_BASEDIR = "/eu/dnetlib/enabling/datasources/";
	private String db;
	private UniqueServiceLocator serviceLocator;
	private ResultSetClientFactory resultSetClientFactory;

	public String findDatasourceId(final String profileId) throws DatasourceManagerServiceException {
		try {
			return serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(
					"/*[.//RESOURCE_IDENTIFIER/@value='" + profileId + "']//EXTRA_FIELDS/FIELD[./key='OpenAireDataSourceId']/value/text()");
		} catch (final Exception e) {
			log.error("Error finding dsId of profile " + profileId, e);
			throw new DatasourceManagerServiceException("Error finding dsId of profile " + profileId, e);
		}
	}

	public String getDatasourceProfile(final String dsId) throws DatasourceManagerServiceException {
		try {
			return serviceLocator.getService(ISLookUpService.class)
					.getResourceProfileByQuery(
							"collection('/db/DRIVER/RepositoryServiceResources/RepositoryServiceResourceType')/*[.//EXTRA_FIELDS/FIELD[./key='OpenAireDataSourceId']/value/text() = '"
									+ dsId + "']");
		} catch (final Exception e) {
			return null;
		}
	}

	public boolean deleteProfile(final String dsId) throws DatasourceManagerServiceException {
		try {
			final SAXReader reader = new SAXReader();

			final String profile = getDatasourceProfile(dsId);

			if (profile != null) {
				final Document docOld = reader.read(new StringReader(profile));
				final String profId = docOld.valueOf("//RESOURCE_IDENTIFIER/@value");
				serviceLocator.getService(ISRegistryService.class).deleteProfile(profId);
			}
			return true;
		} catch (final Exception e) {
			log.error("Error deleting profile", e);
			throw new DatasourceManagerServiceException("Error deleting profile", e);
		}
	}

	public boolean regenerateProfile(final String dsId) throws DatasourceManagerServiceException {

		try {
			final SAXReader reader = new SAXReader();

			final List<String> list = getTransformedDatasourcesByCondition("ds.id= '" + dsId + "'", new ApplyXslt(xslt));

			if (list.size() != 1) { throw new DatasourceManagerServiceException("Illegal number of datasource with id " + dsId + ", size: " + list.size()); }

			final Document doc = reader.read(new StringReader(list.get(0)));

			final ISRegistryService registry = serviceLocator.getService(ISRegistryService.class);

			final String profile = getDatasourceProfile(dsId);

			if (profile != null) {
				final Document docOld = reader.read(new StringReader(profile));
				final String profId = docOld.valueOf("//RESOURCE_IDENTIFIER/@value");

				doc.selectSingleNode("//RESOURCE_IDENTIFIER/@value").setText(profId);

				registry.updateProfile(profId, doc.asXML(), REPOSITORY_SERVICE_RESOURCE_TYPE);
				log.info("Profile " + profId + " UPDATED for ds " + dsId);
			} else {
				final String profId = registry.registerProfile(doc.asXML());
				log.info("Valid Profile " + profId + " REGISTERED for ds " + dsId);
			}
			return true;
		} catch (final Exception e) {
			log.error("Error saving profile, id: " + dsId, e);
			throw new DatasourceManagerServiceException("Error regenerating profile", e);
		}
	}

	public Iterable<String> searchSQL(final String sql) throws DatabaseException {
		final W3CEndpointReference epr = serviceLocator.getService(DatabaseService.class).searchSQL(getDb(), sql);
		return resultSetClientFactory.getClient(epr);
	}

	public boolean updateSQL(final String dsId, final String sql, final boolean delete, final boolean updateprofile) throws DatasourceManagerServiceException {
		log.debug("Executing query SQL: " + sql);

		try {
			serviceLocator.getService(DatabaseService.class).updateSQL(getDb(), sql);
		} catch (final DatabaseException e) {
			throw new DatasourceManagerServiceException(e);
		}

		if (updateprofile) {
			if (delete) {
				return deleteProfile(dsId);
			} else {
				return regenerateProfile(dsId);
			}
		}
		return false;
	}

	public boolean updateSQL(final String dsId, final String sqlTemplate, final Map<String, Object> params, final boolean delete, final boolean updateProfile)
			throws DatasourceManagerServiceException {

		verifyParams(params.values());
		verifyParams(params.keySet().toArray());

		try {
			final Resource resource = new ClassPathResource(TMPLS_BASEDIR + sqlTemplate);
			final StringTemplate st = new StringTemplate(IOUtils.toString(resource.getInputStream()));
			st.setAttributes(params);
			return updateSQL(dsId, st.toString(), delete, updateProfile);
		} catch (final Exception e) {
			log.error("Error in updateSQL", e);
			throw new DatasourceManagerServiceException("Error in updateSQL", e);
		}
	}

	public List<DatasourceDesc> getDatasourcesByCondition(final String condition) throws DatasourceManagerServiceException {
		final SAXReader reader = new SAXReader();
		final List<DatasourceDesc> list = Lists.newArrayList();
		try {
			for (final String s : getXmlDatasourcesByCondition(condition)) {
				final Document doc = reader.read(new StringReader(s));
				list.add(DatasourceFunctions.xmlToDatasourceDesc(doc));
			}
		} catch (final Exception e) {
			log.error("Error obtaining datasources from db", e);
			throw new DatasourceManagerServiceException("Error obtaining datasources from db", e);
		}
		return list;

	}

	private void verifyParams(final Object... params) throws DatasourceManagerServiceException {

		final Pattern pattern = Pattern.compile("\\n");

		for (final Object p : params) {
			log.debug("TESTING SQL PARAM:" + p);
			if ((p == null) || p.toString().isEmpty()) {
				log.error("Parameter null or empty");
				throw new DatasourceManagerServiceException("Parameter null or empty");
			} else if (pattern.matcher(p.toString()).matches()) {
				log.error("Parameter [" + p + "] contains an invalid character");
				throw new DatasourceManagerServiceException("Parameter [" + p + "] contains an invalid character");
			} else {
				log.debug("TEST OK");
			}
		}
	}

	private List<String> getTransformedDatasourcesByCondition(final String condition, final UnaryFunction<String, String> function)
			throws DatasourceManagerServiceException {
		final List<String> list = Lists.newArrayList();
		try {
			for (final String s : getXmlDatasourcesByCondition(condition)) {
				list.add(function.evaluate(s));
			}
		} catch (final Exception e) {
			log.error("Error obtaining datasources from db", e);
			throw new DatasourceManagerServiceException("Error obtaining datasources from db", e);
		}
		return list;
	}

	private Iterable<String> getXmlDatasourcesByCondition(final String condition) throws DatasourceManagerServiceException {
		try {
			final Map<String, Object> params = Maps.newHashMap();

			if ((condition != null) && !condition.trim().isEmpty()) {
				params.put("condition", condition);
			}
			return searchSQL("getDatasources.sql.st", params);
		} catch (final Exception e) {
			log.error("Error obtaining datasources from db", e);
			throw new DatasourceManagerServiceException("Error obtaining datasources from db", e);
		}
	}

	public Iterable<String> searchSQL(final String sqlTemplate, final Map<String, Object> params) throws DatasourceManagerServiceException {
		try {
			final Resource resource = new ClassPathResource(TMPLS_BASEDIR + sqlTemplate);
			final StringTemplate st = new StringTemplate(IOUtils.toString(resource.getInputStream()));
			if (params != null) {
				st.setAttributes(params);
			}

			final String sql = st.toString();

			log.debug("Executing SQL: " + sql);

			return searchSQL(sql);
		} catch (final Exception e) {
			log.error("Error executing sql", e);

			throw new DatasourceManagerServiceException("Error obtaining datasources from db", e);
		}
	}

	public boolean isDefinedParam(final String ifaceId, final String field) throws DatasourceManagerServiceException {
		final Map<String, Object> params = Maps.newHashMap();
		params.put("ifaceId", DatasourceFunctions.asSqlValue(ifaceId));
		params.put("field", DatasourceFunctions.asSqlValue(field));

		final List<String> list = Lists.newArrayList(searchSQL("searchApiCollectionParam.sql.st", params));

		return !list.isEmpty();
	}

	public Date findNextScheduledExecution(final String dsId, final String ifaceId) throws DatasourceManagerServiceException {
		final String xquery = "/*[.//DATAPROVIDER/@interface='" + ifaceId + "' and .//SCHEDULING/@enabled='true']//CRON/text()";
		try {
			final String cronExpression = serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(xquery);
			final CronExpression cron = new CronExpression(cronExpression);
			return cron.getNextValidTimeAfter(new Date());
		} catch (final ISLookUpDocumentNotFoundException e) {
			// When the value is not found a null value must be returned
			return null;
		} catch (final ISLookUpException e) {
			log.error("Error in xquery: " + xquery, e);
			throw new DatasourceManagerServiceException("Error in xquery: " + xquery, e);
		} catch (final ParseException e) {
			log.error("Error parsing cron expression", e);
			throw new DatasourceManagerServiceException("Error parsing cron expression", e);
		}
	}

	public String getDb() {
		return db;
	}

	@Required
	public void setDb(final String db) {
		this.db = db;
	}

	public ResultSetClientFactory getResultSetClientFactory() {
		return resultSetClientFactory;
	}

	@Required
	public void setResultSetClientFactory(final ResultSetClientFactory resultSetClientFactory) {
		this.resultSetClientFactory = resultSetClientFactory;
	}

	public UniqueServiceLocator getServiceLocator() {
		return serviceLocator;
	}

	@Required
	public void setServiceLocator(final UniqueServiceLocator serviceLocator) {
		this.serviceLocator = serviceLocator;
	}

}
