package eu.dnetlib.iis.export.actionmanager.module;

import java.util.Arrays;
import java.util.List;

import org.apache.hadoop.conf.Configuration;

import eu.dnetlib.actionmanager.actions.AtomicAction;
import eu.dnetlib.actionmanager.common.Agent;
import eu.dnetlib.data.mapreduce.util.OafDecoder;
import eu.dnetlib.data.proto.KindProtos.Kind;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.OafProtos.OafRel;
import eu.dnetlib.data.proto.RelTypeProtos.RelType;
import eu.dnetlib.data.proto.RelTypeProtos.SubRelType;
import eu.dnetlib.data.proto.ResultResultProtos.ResultResult;
import eu.dnetlib.iis.websiteusage.schemas.DocumentsWithWebsiteUsageSimilarities;

/**
 * {@link DocumentsWithWebsiteUsageSimilarities} based action builder module.
 * @author mhorst
 *
 */
public class DocumentsWithWebsiteUsageSimilaritiesActionBuilderModuleFactory 
	implements ActionBuilderFactory<DocumentsWithWebsiteUsageSimilarities> {

	private static final AlgorithmName algorithmName = AlgorithmName.document_similarities_websiteusage;
	
	class DocumentsWithWebsiteUsageSimilaritiesActionBuilder extends
	AbstractBuilderModule implements ActionBuilderModule<DocumentsWithWebsiteUsageSimilarities> {
	
		/**
		 * Default similarity level.
		 */
		private final float defaultSimilarityLevel = 0;
		
		/**
		 * Default constructor.
		 * @param predefinedTrust
		 * @param actionSetId
		 */
		public DocumentsWithWebsiteUsageSimilaritiesActionBuilder(
				String predefinedTrust) {
			super(predefinedTrust, algorithmName);
		}
	
		@Override
		public List<AtomicAction> build(
				DocumentsWithWebsiteUsageSimilarities object, Agent agent, 
				String actionSetId) {
			String currentSimDocId = object.getOtherDocumentId().toString();
			String docId = object.getDocumentId().toString();
			Oaf similarityRel = buildSimilarityRel(
					docId, currentSimDocId,
					object.getCovisitedSimilarity()!=null?
							object.getCovisitedSimilarity():
								defaultSimilarityLevel);
			Oaf reverseSimilarityRel = buildSimilarityRel(
					currentSimDocId, docId,
					object.getCovisitedSimilarity()!=null?
							object.getCovisitedSimilarity():
								defaultSimilarityLevel);
			return Arrays.asList(new AtomicAction[] {
					actionFactory.createAtomicAction(
						actionSetId, agent, docId, 
						OafDecoder.decode(similarityRel).getCFQ(), 
						currentSimDocId, 
						similarityRel.toByteArray()),
	//		adding reverse relation
					actionFactory.createAtomicAction(
							actionSetId, agent, currentSimDocId,  
							OafDecoder.decode(reverseSimilarityRel).getCFQ(), 
						docId,
						reverseSimilarityRel.toByteArray())
			});
		}
	
		/**
		 * Builds OAF object.
		 * @param source
		 * @param target
		 * @param score
		 * @return OAF object
		 */
		private Oaf buildSimilarityRel(String source, String target, float score) {
			OafRel.Builder relBuilder = OafRel.newBuilder();
			relBuilder.setSource(source);
			relBuilder.setTarget(target);
			relBuilder.setChild(false);
			relBuilder.setRelType(RelType.resultResult);
			relBuilder.setSubRelType(SubRelType.similarity);
//			TODO shouldn't we have dedicated website usage similarity class
//			TODO we should set it in similaritymetadata field as well
//			relBuilder.setRelClass(value);
			ResultResult.Builder resResBuilder = ResultResult.newBuilder();
			resResBuilder.setSimilarity(ResultResult.Similarity.newBuilder().setType(
					ResultResult.Similarity.Type.WEBUSAGE).setSimilarity(
					score).build());
			relBuilder.setResultResult(resResBuilder.build());
			eu.dnetlib.data.proto.OafProtos.Oaf.Builder oafBuilder = Oaf.newBuilder();
			oafBuilder.setKind(Kind.relation);
			oafBuilder.setRel(relBuilder.build());
			oafBuilder.setDataInfo(buildInference());
			oafBuilder.setTimestamp(System.currentTimeMillis());
			return oafBuilder.build();
		}
		
		@Override
		public AlgorithmName getAlgorithName() {
			return algorithmName;
		}
	}


	@Override
	public ActionBuilderModule<DocumentsWithWebsiteUsageSimilarities> instantiate(
			String predefinedTrust, Configuration config) {
		return new DocumentsWithWebsiteUsageSimilaritiesActionBuilder(
				predefinedTrust);
	}
}
