package eu.dnetlib.app.directindex;

import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.PostConstruct;

import org.apache.commons.lang3.StringUtils;
import org.apache.maven.model.Model;
import org.apache.maven.model.io.xpp3.MavenXpp3Reader;
import org.codehaus.plexus.util.xml.pull.XmlPullParserException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springdoc.core.GroupedOpenApi;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.autoconfigure.solr.SolrAutoConfiguration;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.context.annotation.Bean;
import org.springframework.core.io.ClassPathResource;
import org.springframework.scheduling.annotation.EnableScheduling;

import io.micrometer.core.instrument.ImmutableTag;
import io.micrometer.core.instrument.Metrics;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.info.Info;
import io.swagger.v3.oas.models.info.License;
import io.swagger.v3.oas.models.servers.Server;

@EnableCaching
@EnableScheduling
@SpringBootApplication(exclude = { SolrAutoConfiguration.class })
public class DirectIndexApplication {

	private static final Logger log = LoggerFactory.getLogger(DirectIndexApplication.class);

	private static final String DIRECT_INDEX_DESC = "APIs documentation";

	private static final License AGPL_3_LICENSE =
			new License().name("GNU Affero General Public License v3.0 or later").url("https://www.gnu.org/licenses/agpl-3.0.txt");

	@Value("${maven.pom.path}")
	private ClassPathResource pom;

	@Value("${dnet.directindex.baseurl}")
	private String publicUrl;

	@Value("${dnet.directindex.title}")
	private String publicTitle;

	@Value("${dnet.directindex.description}")
	private String publicDesc;

	public static void main(final String[] args) {
		SpringApplication.run(DirectIndexApplication.class, args);
	}

	@PostConstruct
	public void init() {
		final MavenXpp3Reader reader = new MavenXpp3Reader();
		try {
			final Model model = reader.read(new InputStreamReader(pom.getInputStream()));

			log.info(String.format("registering metric for %s", model.getArtifactId()));

			final ImmutableTag tag1 = new ImmutableTag("component", model.getGroupId() + ":" + model.getArtifactId());
			final ImmutableTag tag2 = new ImmutableTag("version", model.getVersion());
			final ImmutableTag tag3 = new ImmutableTag("scmtag", model.getScm().getTag());

			Metrics.gauge("micrometer_info", Arrays.asList(tag1, tag2, tag3), 1);
		} catch (IOException | XmlPullParserException e) {
			log.error("Error registering metric", e);
		}
	}

	@Bean
	public OpenAPI newSwaggerDocket() {
		final List<Server> servers = new ArrayList<>();
		if (StringUtils.isNotBlank(publicUrl)) {
			final Server server = new Server();
			server.setUrl(publicUrl);
			server.setDescription(publicDesc);
			servers.add(server);
		}

		return new OpenAPI()
				.servers(servers)
				.info(new Info()
						.title(publicTitle)
						.description(DIRECT_INDEX_DESC)
						.version(currentVersion())
						.license(AGPL_3_LICENSE))
				.tags(new ArrayList<>());
	}

	private String currentVersion() {
		try {
			return new MavenXpp3Reader().read(new InputStreamReader(pom.getInputStream())).getVersion();
		} catch (IOException | XmlPullParserException e) {
			return "UNKNOWN";
		}
	}

	@Bean
	public GroupedOpenApi publicApi() {
		return GroupedOpenApi.builder()
				.group(publicTitle)
				.pathsToMatch("/api/**")
				.build();
	}

}
