package eu.dnetlib.app.directindex.solr;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.impl.CloudSolrClient;
import org.apache.solr.client.solrj.impl.Http2SolrClient;
import org.apache.solr.client.solrj.impl.LBHttp2SolrClient;
import org.apache.solr.client.solrj.impl.LBHttpSolrClient;
import org.apache.solr.client.solrj.impl.LBSolrClient.Endpoint;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

@Component
public class SolrIndexClientFactory {

	private static final Log log = LogFactory.getLog(SolrIndexClientFactory.class);

	public static final String CHROOT_SEPARATOR = "/";

	@Value("${dnet.directindex.solr.urls}")
	public String solrUrls;

	@Value("${dnet.directindex.solr.collection}")
	public String solrCollection;

	@Value("${dnet.directindex.solr.client}")
	public String solrClientType;

	@Value("${dnet.directindex.solr.zk_chroot}")
	public String solrZkChroot;

	public SolrIndexClient getClient() {

		final SolrClient client;

		switch (solrClientType.toUpperCase()) {
		case "LBHTTP1":
			client = getLoadBalancedHttp1Client(solrUrls, solrCollection);
			break;
		case "LBHTTP2":
			client = getLoadBalancedHttp2Client(solrUrls, solrCollection);
			break;
		case "CLOUD":
			client = getCloudClient(solrUrls, solrCollection);
			break;
		case "ZK_CLOUD":
			client = getZKCloudClient(solrUrls, solrCollection, Optional.of(solrZkChroot));
			break;
		default:
			client = getSimpleClient(solrUrls, solrCollection);
			break;
		}

		return new SolrIndexClient(client);
	}

	@Deprecated
	protected SolrClient getLoadBalancedHttp1Client(final String urls, final String coll) {
		log.warn(String.format("Initializing deprecated solr client (Load Balanced HTTP/1), urls: %s, collection: %s", urls, coll));

		final String[] endpoints = Arrays.stream(urls.split(","))
				.map(String::trim)
				.filter(StringUtils::isNotBlank)
				.map(url -> url += "/" + coll)
				.toArray(String[]::new);

		return new LBHttpSolrClient.Builder()
				.withHttpClient(new DefaultHttpClient())
				.withBaseEndpoints(endpoints)
				.build();

	}

	protected SolrClient getLoadBalancedHttp2Client(final String urls, final String coll) {
		log.debug(String.format("Initializing solr client (Load Balanced HTTP/2), urls: %s, collection: %s", urls, coll));

		final Endpoint[] endpoints = Arrays.stream(urls.split(","))
				.map(String::trim)
				.filter(StringUtils::isNotBlank)
				.map(url -> url += "/" + coll)
				.map(Endpoint::new)
				.toArray(Endpoint[]::new);

		final Http2SolrClient http2SolrClient = new Http2SolrClient.Builder().build();

		return new LBHttp2SolrClient.Builder(http2SolrClient, endpoints).build();
	}

	private SolrClient getCloudClient(final String urls, final String coll) {
		log.debug(String.format("Initializing solr client (cloud), urls: %s, collection: %s", urls, coll));

		final List<String> urlList = Arrays.stream(urls.split(","))
				.map(String::trim)
				.filter(StringUtils::isNotBlank)
				.toList();

		return new CloudSolrClient.Builder(urlList)
				.withParallelUpdates(true)
				.withDefaultCollection(coll)
				.build();
	}

	protected SolrClient getZKCloudClient(final String urls, final String coll, final Optional<String> chroot) {
		log.debug(String.format("Initializing solr client (cloud), urls: %s, collection: %s", urls, coll));

		final List<String> urlList = Arrays.stream(urls.split(","))
				.map(String::trim)
				.filter(StringUtils::isNotBlank)
				.toList();

		return new CloudSolrClient.Builder(urlList, chroot)
				.withParallelUpdates(true)
				.withDefaultCollection(coll)
				.build();

	}

	protected SolrClient getSimpleClient(final String url, final String coll) {
		log.debug(String.format("Initializing solr client (SIMPLE), url: %s, collection: %s", url, coll));
		final Endpoint endpoint = new Endpoint(url.trim() + "/" + coll);
		final Http2SolrClient http2SolrClient = new Http2SolrClient.Builder().build();
		return new LBHttp2SolrClient.Builder(http2SolrClient, endpoint).build();
	}
}
