package eu.dnetlib.dhp.solr.mapping;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import eu.dnetlib.dhp.schema.solr.*;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.solr.common.SolrInputDocument;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static eu.dnetlib.dhp.solr.mapping.MappingUtils.*;

public class SolrInputDocumentMapper implements Serializable {

    private static final String INDEX_JSON_RESULT = "__json";
    private static final String INDEX_RESULT = "__result";

    private static final ObjectMapper mapper = new ObjectMapper()
            .setSerializationInclusion(JsonInclude.Include.NON_EMPTY) ;

    private static final String DELIMITER = "||";
    
    private static final int MAX_URL_LENGTH = 1000;
    private static final long MAX_URLS = 32;

    public static SolrInputDocument map(SolrRecord sr, String xml, boolean shouldFilterXmlPayload) throws JsonProcessingException {

        final SolrInputDocument doc = new SolrInputDocument();

        if (StringUtils.isNotBlank(xml) && !filterXmlPayload(sr, shouldFilterXmlPayload)) {
            doc.addField(INDEX_RESULT, xml);
        }
        if (Objects.isNull(sr)) {
            throw new IllegalArgumentException("SolrRecord cannot be null");
        }

        doc.addField(INDEX_JSON_RESULT, mapper.writeValueAsString(sr));

        doc.addField("__indexrecordidentifier", sr.getHeader().getId());

        mapCommonFields(doc, sr);
        addRelatedRecordFields(doc, Optional.ofNullable(sr.getLinks()).orElse(new ArrayList<>()));

        switch (sr.getHeader().getRecordType()) {
            case publication, dataset, software, other -> mapResultFields(doc, sr.getResult(), sr.getLinks());
            case datasource -> mapDatasourceFields(doc, sr.getDatasource(), sr.getLinks());
            case organization -> mapOrganizationFields(doc, sr.getOrganization(), filterLinks(sr.getLinks(), "merges"));
            case project -> mapProjectFields(doc, sr.getProject());
            case person -> {
            }
            default -> throw new IllegalStateException("Unexpected value: " + sr.getHeader().getRecordType());
        }

        return doc;
    }

    private static boolean filterXmlPayload(SolrRecord sr, Boolean shouldFilterXmlPayload) {
        if (Boolean.TRUE.equals(shouldFilterXmlPayload)) {

            boolean isProject = RecordType.project.equals(sr.getHeader().getRecordType());
            boolean isRelatedToEcFunding = Optional
                    .ofNullable(sr.getLinks())
                    .map(
                            links -> links
                                    .stream()
                                    .anyMatch(
                                            rr -> {
                                                final boolean isRelatedProject = RecordType.project
                                                        .equals(rr.getHeader().getRelatedRecordType());
                                                final String funderShortName = Optional
                                                        .ofNullable(rr.getFunding())
                                                        .map(Funding::getFunder)
                                                        .map(Funder::getShortname)
                                                        .orElse("");
                                                return isRelatedProject && "EC".equals(funderShortName);
                                            }))
                    .orElse(false);
            return !(isProject || isRelatedToEcFunding);
        }
        return false;
    }

    private static void mapCommonFields(SolrInputDocument doc, SolrRecord sr) {
        // objIdentifier xpath="//header/dri:objIdentifier"
        doc.addField("objidentifier", sr.getHeader().getId());

        // oaftype value="local-name(//*[local-name()='entity']/*[local-name() != 'extraInfo'])
        doc.addField("oaftype", mapToOafType(sr.getHeader().getRecordType()));

        // collectedfromname xpath="distinct-values(
        //      *[local-name()='entity']/*/*[local-name()='collectedfrom']/@name |
        //      *[local-name()='entity']/*//*[local-name() = 'instance']/*[local-name()='collectedfrom']/@name)"
        addField("collectedfromname", Optional
                .ofNullable(sr.getCollectedfrom())
                .map(c -> c.stream()
                        .map(Provenance::getDsName)
                        .toList())
                .orElse(new ArrayList<>()), doc);

        // collectedfromdatasourceid xpath="distinct-values(
        //      *[local-name()='entity']/*/*[local-name()='collectedfrom']/@id |
        //      *[local-name()='entity']/*//*[local-name() = 'instance']/*[local-name()='collectedfrom']/@id)"
        addField("collectedfromdatasourceid", Optional.ofNullable(sr.getCollectedfrom())
                .stream()
                .flatMap(Collection::stream)
                .map(Provenance::getDsId)
                .toList(), doc);

        // influence" xpath="//measure[@id='influence']/@score/number()
        addField("influence", getMeasure(sr, "influence", "score"), doc);

        // influence_class type="string" xpath="//measure[@id='influence']/@class/string()"
        addField("influence_class", getMeasure(sr, "influence", "class"), doc);

        // popularity xpath="//measure[@id='popularity']/@score/string()"
        addField("popularity", getMeasure(sr, "popularity", "score"), doc);

        // popularity_class xpath="//measure[@id='popularity']/@class/string()"
        addField("popularity_class", getMeasure(sr, "popularity", "class"), doc);

        // citation_count xpath="//measure[@id='influence_alt']/@score/number()"
        addField("citation_count", getMeasure(sr, "influence_alt", "score"), doc);

        // citation_count_class xpath="//measure[@id='influence_alt']/@class/string()"
        addField("citation_count_class", getMeasure(sr, "influence_alt", "class"), doc);

        // popularity_alt xpath="//measure[@id='popularity_alt']/@score/number()"
        addField("popularity_alt", getMeasure(sr, "popularity_alt", "score"), doc);

        // popularity_alt_class xpath="//measure[@id='popularity_alt']/@class/string()"
        addField("popularity_alt_class", getMeasure(sr, "popularity_alt", "class"), doc);

        // impulse xpath="//measure[@id='impulse']/@score/number()"
        addField("impulse", getMeasure(sr, "impulse", "score"), doc);

        // impulse_class xpath="//measure[@id='impulse']/@class/string()"
        addField("impulse_class", getMeasure(sr, "impulse", "class"), doc);


        // dateofcollection value="//header/*[local-name()='dateOfCollection']"
        // TODO addField("dateofcollection", sr.getHeader().getDateOfCollection(), doc);

        // status xpath="//header/*[local-name()='status']"/>
        addField("status", Optional.ofNullable(sr.getHeader().getStatus()).map(Enum::toString), doc);

        // originalid xpath="//*[local-name()='entity']/*/*[local-name()='originalId']"
        addField("originalid", sr.getHeader().getOriginalId(), doc);

        // pid xpath="//*[local-name()='entity']/*/pid/text()"
        addField("pid", Optional.ofNullable(sr.getPid())
                .stream()
                .flatMap(p -> p.stream().map(Pid::getValue))
                .toList(), doc);

        // pidclassid xpath="distinct-values(//*[local-name()='entity']/*/pid/@classid)"
        addField("pidclassid", Optional.ofNullable(sr.getPid())
                .stream()
                .flatMap(p -> p.stream().map(Pid::getTypeCode))
                .toList(), doc);

        // deletedbyinference xpath="//*[local-name()='entity']//datainfo/deletedbyinference"
        addField("deletedbyinference", sr.getHeader().getDeletedbyinference().toString(), doc);

        // provenanceactionclassid xpath="//*[local-name()='entity']//datainfo/provenanceaction/@classid"
        // TODO addField("provenanceactionclassid", sr.getHeader().getProvenanceactionclassid(), doc);

        // contextid xpath="distinct-values(//*[local-name()='entity']/*/context/@id)"
        addField("contextid", Optional.ofNullable(sr.getContext())
                .stream()
                .flatMap(Collection::stream)
                .map(Context::getId)
                .toList(), doc);

        // contextname xpath="distinct-values(//*[local-name()='entity']/*/context/@label)"
        addField("contextname", Optional.ofNullable(sr.getContext())
                .stream()
                .flatMap(Collection::stream)
                .map(Context::getLabel)
                .toList(), doc);

        // community
        //      xpath="//*[local-name()='entity']/*/context[@type='community' or @type='ri']"/>
        //      value="distinct-values(concat(@id, '||', @label))"
        addField("community", Optional.ofNullable(sr.getContext())
                .stream()
                .flatMap(Collection::stream)
                .filter(c -> "community".equals(c.getType()) || "ri".equals(c.getType()))
                .map(c -> new StringJoiner(DELIMITER)
                        .add(c.getId())
                        .add(c.getLabel())
                        .toString())
                .toList(), doc);

        // communityid xpath="distinct-values(//*[local-name()='entity']/*/context[@type='community' or @type='ri']/@id)"
        addField("communityid", Optional.ofNullable(sr.getContext())
                .stream()
                .flatMap(Collection::stream)
                .filter(c -> "community".equals(c.getType()) || "ri".equals(c.getType()))
                .map(Context::getId)
                .toList(), doc);

        // categoryid xpath="distinct-values(//*[local-name()='entity']/*/context/category/@id)"
        addField("categoryid", Optional.ofNullable(sr.getContext())
                        .stream()
                        .flatMap(Collection::stream)
                        .flatMap(c -> Optional.ofNullable(c.getCategory())
                                .stream()
                                .flatMap(Collection::stream)
                                .map(Category::getId))
                        .toList(), doc);

        // conceptname xpath="distinct-values(//*[local-name()='entity']/*/context/category//concept/@label)"/>
        addField("conceptname", Optional.ofNullable(sr.getContext())
                .stream()
                .flatMap(Collection::stream)
                .flatMap(c -> Optional.ofNullable(c.getCategory())
                        .stream()
                        .flatMap(Collection::stream)
                        .flatMap(ca -> Optional.ofNullable(ca.getConcept())
                                .stream()
                                .flatMap(Collection::stream)
                                .map(Concept::getLabel)))
                        .toList(), doc);
    }

    private static Optional<String> getMeasure(SolrRecord sr, String id, String unitCode) {

        return Optional.ofNullable(sr.getMeasures())
                .stream()
                .flatMap(measures -> measures
                    .stream()
                    .filter(m -> id.equals(m.getId()))
                    .flatMap(m -> m.getUnit().stream())
                    .filter(u -> unitCode.equals(u.getCode()))
                    .map(CodeLabel::getLabel))
                .findFirst();
    }

    private static void addRelatedRecordFields(SolrInputDocument doc, List<RelatedRecord> links) {

        // reldatasourcecompatibilityid xpath="distinct-values(//*[local-name()='entity']/*//rel[./to/@type='datasource']/openairecompatibility/@classid)";
        addField("reldatasourcecompatibilityid", links
                .stream()
                .filter(rr -> RecordType.datasource.equals(rr.getHeader().getRelatedRecordType()))
                .map(RelatedRecord::getOpenairecompatibility)
                .map(CodeLabel::getCode)
                .toList(), doc);

        // relproject value="distinct-values(concat(./text(), '||', dnet:pickFirst(../acronym/text(), ../title/text())))" xpath="//*[local-name()='entity']/*//rel/to[@type='project']"
        addField("relproject", links
                .stream()
                .filter(rr -> RecordType.project.equals(rr.getHeader().getRelatedRecordType()))
                .map(rr -> {
                    final String projectName = ObjectUtils.firstNonNull(rr.getAcronym(), rr.getProjectTitle());
                    if (StringUtils.isNotBlank(projectName)) {
                        return new StringJoiner(DELIMITER)
                                .add(rr.getHeader().getRelatedIdentifier())
                                .add(projectName)
                                .toString();
                    }
                    return null;
                })
                .filter(Objects::nonNull)
                .toList(), doc);

        // relprojectid xpath="distinct-values(//*[local-name()='entity']/*//rel/to[@type='project'])"
        addField("relprojectid", links
                .stream()
                .map(RelatedRecord::getHeader)
                .filter(header -> RecordType.project.equals(header.getRelatedRecordType()))
                .map(RelatedRecordHeader::getRelatedIdentifier)
                .toList(), doc);

        // relprojectcode xpath="distinct-values(//*[local-name()='entity']/*//rel[./to/@type='project']/code)"
        addField("relprojectcode", links
                .stream()
                .filter(rr -> RecordType.project.equals(rr.getHeader().getRelatedRecordType()))
                .map(RelatedRecord::getCode)
                .toList(), doc);

        // relprojectname xpath="distinct-values(//*[local-name()='entity']/*//rel[./to/@type='project']/acronym))"
        addField("relprojectname", links
                .stream()
                .filter(rr -> RecordType.project.equals(rr.getHeader().getRelatedRecordType()))
                .map(RelatedRecord::getAcronym)
                .toList(), doc);

        // relprojecttitle xpath="distinct-values(//*[local-name()='entity']/*//rel[./to/@type='project']/title)"
        addField("relprojecttitle", links
                .stream()
                .filter(rr -> RecordType.project.equals(rr.getHeader().getRelatedRecordType()))
                .map(RelatedRecord::getProjectTitle)
                .toList(), doc);

        // relcontracttypename xpath="distinct-values(//*[local-name()='entity']/*//rel[./to/@type='project']/contracttype/@classname)"
        addField("relcontracttypename", links
                .stream()
                .filter(rr -> RecordType.project.equals(rr.getHeader().getRelatedRecordType()))
                .flatMap(rr -> Optional.ofNullable(rr.getContracttype())
                        .map(CodeLabel::getLabel)
                        .stream())
                .toList(), doc);

        // relorganizationcountryid xpath="distinct-values(//*[local-name()='entity']/*//rel[./to/@type='organization']/country/@classid)"
        addField("relorganizationcountryid", links
                .stream()
                .filter(rr -> RecordType.organization.equals(rr.getHeader().getRelatedRecordType()))
                .flatMap(rr -> Optional.ofNullable(rr.getCountry())
                        .map(Country::getCode)
                        .stream())
                .toList(), doc);

        // relorganizationcountryname xpath="distinct-values(//*[local-name()='entity']/*//rel[./to/@type='organization']/country/@classname)"
        addField("relorganizationcountryname", links
                .stream()
                .filter(rr -> RecordType.organization.equals(rr.getHeader().getRelatedRecordType()))
                .flatMap(rr -> Optional.ofNullable(rr.getCountry())
                        .map(Country::getLabel)
                        .stream())
                .toList(), doc);

        // relorganizationid xpath="distinct-values(//*[local-name()='entity']/*//rel/to[@type='organization'])"/>
        addField("relorganizationid", links
                .stream()
                .map(RelatedRecord::getHeader)
                .filter(header -> RecordType.organization.equals(header.getRelatedRecordType()))
                .map(RelatedRecordHeader::getRelatedIdentifier)
                .toList(), doc);

        // relorganizationname xpath="distinct-values(//*[local-name()='entity']/*//rel[./to/@type='organization']/legalname)"
        addField("relorganizationname", links
                .stream()
                .filter(rr -> RecordType.organization.equals(rr.getHeader().getRelatedRecordType()))
                .map(RelatedRecord::getLegalname)
                .toList(), doc);

        // relorganizationshortname xpath="distinct-values(//*[local-name()='entity']/*//rel[./to/@type='organization']/legalshortname)"/>
        addField("relorganizationshortname", links
                .stream()
                .filter(rr -> RecordType.organization.equals(rr.getHeader().getRelatedRecordType()))
                .map(RelatedRecord::getLegalshortname)
                .toList(), doc);

        // relorganization
        //      xpath="//*[local-name()='entity']/*//rel[./to/@type='organization']"
        //      value="distinct-values(concat(./to, '||', ./legalname))"
        addField("relorganization", links
                .stream()
                .filter(rr -> RecordType.organization.equals(rr.getHeader().getRelatedRecordType()))
                .map(rr -> Stream.of(rr.getHeader().getRelatedIdentifier(), rr.getLegalname())
                        .filter(StringUtils::isNotBlank)
                        .collect(Collectors.joining(DELIMITER)))
                .toList(), doc);

        // relresultid xpath="distinct-values(//*[local-name()='entity']/*//rel/to[@type='publication' or @type='dataset' or @type='software' or @type='otherresearchproduct'])"
        addField("relresultid", links
                .stream()
                .map(RelatedRecord::getHeader)
                .filter(rh -> "result".equals(mapToOafType(rh.getRelatedRecordType())))
                .map(RelatedRecordHeader::getRelatedIdentifier)
                .toList(), doc);

        // relresulttype xpath="distinct-values(//*[local-name()='entity']/*//rel/to/@type)"
        addField("relresulttype", links
                .stream()
                .map(RelatedRecord::getHeader)
                .map(RelatedRecordHeader::getRelatedRecordType)
                .map(Enum::toString)
                .toList(), doc);


        // relclass xpath="distinct-values(//*[local-name()='entity']/*//rel/to/@class)"
        addField("relclass", links
                .stream()
                .map(RelatedRecord::getHeader)
                .map(RelatedRecordHeader::getRelationClass)
                .toList(), doc);

        final List<Funding> fundingStream = links
                .stream()
                .filter(rr -> RecordType.project.equals(rr.getHeader().getRelatedRecordType()))
                .map(RelatedRecord::getFunding)
                .filter(Objects::nonNull)
                .toList();

        // relfundinglevel0_id xpath="//*[local-name()='entity']//rel/funding/funding_level_0"
        addField("relfundinglevel0_id", fundingStream
                .stream()
                .map(Funding::getLevel0)
                .filter(Objects::nonNull)
                .map(FundingLevel::getId)
                .toList(), doc);

        // relfundinglevel0_name xpath="//*[local-name()='entity']//rel/funding/funding_level_0/@name/string()"
        addField("relfundinglevel0_name", fundingStream
                .stream()
                .map(Funding::getLevel0)
                .filter(Objects::nonNull)
                .map(FundingLevel::getName)
                .toList(), doc);

        // relfundinglevel1_id xpath="//*[local-name()='entity']//rel/funding/funding_level_1"
        addField("relfundinglevel1_id", fundingStream
                .stream()
                .map(Funding::getLevel1)
                .filter(Objects::nonNull)
                .map(FundingLevel::getId)
                .toList(), doc);

        // relfundinglevel1_name xpath="//*[local-name()='entity']//rel/funding/funding_level_1/@name/string()"
        addField("relfundinglevel1_name", fundingStream
                .stream()
                .map(Funding::getLevel1)
                .filter(Objects::nonNull)
                .map(FundingLevel::getName)
                .toList(), doc);

        // relfundinglevel2_id xpath="//*[local-name()='entity']//rel/funding/funding_level_2"
        addField("relfundinglevel2_id", fundingStream
                .stream()
                .map(Funding::getLevel2)
                .filter(Objects::nonNull)
                .map(FundingLevel::getId)
                .toList(), doc);

        // relfundinglevel2_name xpath="//*[local-name()='entity']//rel/funding/funding_level_2/@name/string()"
        addField("relfundinglevel2_name", fundingStream
                .stream()
                .map(Funding::getLevel2)
                .filter(Objects::nonNull)
                .map(FundingLevel::getName)
                .toList(), doc);

        // relfunder
        //      xpath="//*[local-name()='entity']//rel/funding/funder"
        //      value="distinct-values(concat(@id, '||', @name, '||', @shortname))"
        addField("relfunder", links
                .stream()
                .filter(rr -> RecordType.project.equals(rr.getHeader().getRelatedRecordType()))
                .flatMap(rr -> Optional.ofNullable(rr.getFunding())
                        .flatMap(funding -> Optional.ofNullable(funding.getFunder())
                                .map(f -> Stream.of(f.getId(), f.getName(), f.getShortname())
                                                .filter(StringUtils::isNotBlank)
                                                .collect(Collectors.joining(DELIMITER))))
                        .stream())
                .toList(), doc);

        // relfunderid xpath="distinct-values(//*[local-name()='entity']//rel/funding/funder/@id)
        addField("relfunderid", fundingStream
                .stream()
                .map(Funding::getFunder)
                .filter(Objects::nonNull)
                .map(Funder::getId)
                .toList(), doc);

        // relfundershortname xpath="distinct-values(//*[local-name()='entity']//rel/funding/funder/@shortname)
        addField("relfundershortname", fundingStream
                .stream()
                .map(Funding::getFunder)
                .filter(Objects::nonNull)
                .map(Funder::getShortname)
                .toList(), doc);

        // semrelid
        //      xpath="//*[local-name()='entity']//rel"
        //      value="concat(./to/text(), '||', ./to/@class/string())"
        addField("semrelid", links
                .stream()
                .map(RelatedRecord::getHeader)
                .map(rh -> new StringJoiner(DELIMITER)
                        .add(rh.getRelatedIdentifier())
                        .add(rh.getRelationClass())
                        .toString())
                .toList(), doc);
    }

    private static List<RelatedRecord> filterLinks(List<RelatedRecord> links, String... filter) {
        HashSet<String> filterSet = Stream.of(filter).collect(Collectors.toCollection(HashSet::new));
        if (Objects.isNull(links)) {
            return new ArrayList<>();
        }
        return links
                .stream()
                .filter(l -> filterSet.contains(l.getHeader().getRelationClass()))
                .toList();
    }

    private static void mapDatasourceFields(SolrInputDocument doc, Datasource d, List<RelatedRecord> links) {

        // datasourceofficialname xpath="//*[local-name()='entity']/*[local-name()='datasource']/officialname"
        addField("datasourceofficialname", d.getOfficialname(), doc);

        // datasourceenglishname xpath="//*[local-name()='entity']/*[local-name()='datasource']/englishname"
        addField("datasourceenglishname", d.getEnglishname(), doc);

        // datasourceoddescription xpath="//*[local-name()='entity']/*[local-name()='datasource']/oddescription"
        addField("datasourceoddescription", d.getDescription(), doc);

        //TODO datasourceodsubjects xpath="//*[local-name()='entity']/*[local-name()='datasource']/odsubjects"/>
        //addField("datasourceodsubjects", d.getDatasourceodsubjects(), doc);

        // datasourceodlanguages xpath="//*[local-name()='entity']/*[local-name()='datasource']/odlanguages"
        addField("datasourceodlanguages", d.getLanguages(), doc);

        // datasourceodcontenttypes xpath="//*[local-name()='entity']/*[local-name()='datasource']/odcontenttypes"
        addField("datasourceodcontenttypes", d.getOdcontenttypes(), doc);

        // datasourcetypename xpath="//*[local-name()='entity']/*[local-name()='datasource']/datasourcetype/@classname"
        addField("datasourcetypename", Optional.ofNullable(d.getDatasourcetype()).map(CodeLabel::getLabel), doc);

        // datasourcetypeuiid xpath="//*[local-name()='entity']/*[local-name()='datasource']/datasourcetypeui/@classid"
        addField("datasourcetypeuiid", Optional.ofNullable(d.getDatasourcetypeui()).map(CodeLabel::getCode), doc);

        // datasourcetypeuiname xpath="//*[local-name()='entity']/*[local-name()='datasource']/datasourcetypeui/@classname"
        addField("datasourcetypeuiname", Optional.ofNullable(d.getDatasourcetypeui()).map(CodeLabel::getLabel), doc);

        // datasourcecompatibilityid xpath="//*[local-name()='entity']/*[local-name()='datasource']/openairecompatibility/@classid"
        addField("datasourcecompatibilityid", Optional.ofNullable(d.getOpenairecompatibility()).map(CodeLabel::getCode), doc);

        // datasourcecompatibilityname xpath="//*[local-name()='entity']/*[local-name()='datasource']/openairecompatibility/@classname"
        addField("datasourcecompatibilityname", Optional.ofNullable(d.getOpenairecompatibility()).map(CodeLabel::getLabel), doc);

        // datasourcesubject" xpath="//*[local-name()='entity']/*[local-name()='datasource']/subjects"
        addField("datasourcesubject", Optional.ofNullable(d.getSubjects())
                .map(s -> s.stream().map(Subject::getValue).toList())
                .orElse(new ArrayList<>()), doc);

        // datasourcejurisdiction xpath="//*[local-name()='entity']/*[local-name()='datasource']/jurisdiction/@classname"
        addField("datasourcejurisdiction", Optional.ofNullable(d.getJurisdiction()).map(CodeLabel::getLabel), doc);

        // datasourcethematic xpath="//*[local-name()='entity']/*[local-name()='datasource']/thematic"
        addField("datasourcethematic", Optional.ofNullable(d.getThematic()).map(Object::toString), doc);

        // eosctype xpath="//*[local-name()='entity']/*[local-name()='datasource']/eosctype/@classname"
        addField("eosctype", Optional.ofNullable(d.getEosctype()).map(CodeLabel::getLabel), doc);

        // eoscdatasourcetype" xpath="//*[local-name()='entity']/*[local-name()='datasource']/eoscdatasourcetype/@classname"
        addField("eoscdatasourcetype", Optional.ofNullable(d.getEoscdatasourcetype()).map(CodeLabel::getLabel), doc);

        final List<String> orgCountries = filterLinks(links, "isProvidedBy")
                .stream()
                .flatMap(rr -> Optional.ofNullable(rr.getCountry()).map(Country::getCode).stream())
                .toList();
        // country xpath="distinct-values(
        //      *[local-name()='entity']/*/country/@classid |
        //      *[local-name()='entity']/*//rel[./to/@type='organization']/country/@classid |
        //      *[local-name()='entity']//funder/@jurisdiction)"
        addField("country", orgCountries, doc);

        // countrynojurisdiction xpath="distinct-values(
        //      *[local-name()='entity']/*/country/@classid |
        //      *[local-name()='entity']/*//rel[./to/@type='organization']/country/@classid)"
        addField("countrynojurisdiction", orgCountries, doc);

    }

    private static void mapOrganizationFields(SolrInputDocument doc, Organization o, List<RelatedRecord> links) {

        // organizationdupid xpath="//*[local-name()='entity']/*//children/organization/@objidentifier"
        addField("organizationdupid", links.stream()
                .map(RelatedRecord::getHeader)
                .map(RelatedRecordHeader::getRelatedIdentifier)
                .toList(), doc);

        // organizationlegalshortname xpath="distinct-values(//*[local-name()='entity']/*[local-name()='organization']//legalshortname)"
        addField("organizationlegalshortname",
                Collections.singletonList(o.getLegalshortname()),
                links.stream().map(RelatedRecord::getLegalshortname).toList(), doc);

        // organizationlegalname xpath="distinct-values(//*[local-name()='entity']/*[local-name()='organization']//legalname)"
        addField("organizationlegalname",
                Collections.singletonList(o.getLegalshortname()),
                links.stream().map(RelatedRecord::getLegalname).toList(), doc);

        // organizationalternativenames xpath="distinct-values(//*[local-name()='entity']/*[local-name()='organization']//alternativeNames)"
        addField("organizationalternativenames", o.getLegalshortname(), doc);

        // country xpath="distinct-values(
        //      *[local-name()='entity']/*/country/@classid |
        //      *[local-name()='entity']/*//rel[./to/@type='organization']/country/@classid |
        //      *[local-name()='entity']//funder/@jurisdiction)"
        addField("country", Optional.ofNullable(o.getCountry()).map(CodeLabel::getCode), doc);

        // countrynojurisdiction xpath="distinct-values(
        //      *[local-name()='entity']/*/country/@classid |
        //      *[local-name()='entity']/*//rel[./to/@type='organization']/country/@classid)"
        addField("countrynojurisdiction", Optional.ofNullable(o.getCountry()).map(CodeLabel::getCode), doc);
    }

    private static void mapProjectFields(SolrInputDocument doc, Project p) {

        // projectcode xpath="//*[local-name()='entity']/*[local-name()='project']/code"
        addField("projectcode", p.getCode(), doc);

        // projectcode_nt xpath="//*[local-name()='entity']/*[local-name()='project']/code"
        addField("projectcode_nt", p.getCode(), doc);

        // projectacronym xpath="//*[local-name()='entity']/*[local-name()='project']/acronym"
        addField("projectacronym", p.getAcronym(), doc);

        // projecttitle xpath="//*[local-name()='entity']/*[local-name()='project']/title"
        addField("projecttitle", p.getTitle(), doc);

        // projecttitle_alternative xpath="//*[local-name()='entity']/*[local-name()='project']/title"
        addField("projecttitle_alternative", p.getTitle(), doc);

        // projectstartdate value="//*[local-name()='entity']/*[local-name()='project']/startdate
        addField("projectstartdate", Optional.ofNullable(p.getStartdate()).map(MappingUtils::normalizeDate), doc);

        // projectstartyear value="dnet:extractYear(//*[local-name()='entity']/*[local-name()='project']/startdate)"
        addField("projectstartyear", Optional.ofNullable(p.getStartdate())
                .map(MappingUtils::normalizeDate)
                .map(MappingUtils::extractYear), doc);

        // projectenddate value="//*[local-name()='entity']/*[local-name()='project']/enddate"
        addField("projectenddate", Optional.ofNullable(p.getEnddate()).map(MappingUtils::normalizeDate), doc);

        // projectendyear value="dnet:extractYear(//*[local-name()='entity']/*[local-name()='project']/enddate)"
        addField("projectendyear", Optional.ofNullable(p.getEnddate())
                .map(MappingUtils::normalizeDate)
                .map(MappingUtils::extractYear), doc);

        // projectcallidentifier xpath="//*[local-name()='entity']/*[local-name()='project']/callidentifier"
        addField("projectcallidentifier", p.getCallidentifier(), doc);

        // projectduration xpath="//*[local-name()='entity']/*[local-name()='project']/duration"
        addField("projectduration", p.getDuration(), doc);

        // projectkeywords xpath="//*[local-name()='entity']/*[local-name()='project']/keywords"
        addField("projectkeywords", p.getKeywords(), doc);

        //TODO  <FIELD indexable="true" multivalued="false" name="projectecsc39" result="false" stat="false" tokenizable="false" xpath="distinct-values(//*[local-name()='entity']/*[local-name()='project']/ecsc39)"/>
        /*
        Optional.ofNullable(p.getProjectEcSc39())
                .ifPresent(val -> doc.addField("projectecsc39", val));
        */

        // projectoamandatepublications xpath="//*[local-name()='entity']/*[local-name()='project']/oamandatepublications"
        addField("projectoamandatepublications", p.getOamandatepublications(), doc);

        Optional<FundingLevel> fundingL0 = Optional.ofNullable(p.getFunding()).map(Funding::getLevel0);

        // fundinglevel0_id xpath="//*[local-name()='entity']/*[local-name()='project']/fundingtree//funding_level_0/id"
        addField("fundinglevel0_id", fundingL0.map(FundingLevel::getId), doc);

        // fundinglevel0_name xpath="//*[local-name()='entity']/*[local-name()='project']/fundingtree//funding_level_0/name"
        addField("fundinglevel0_name", fundingL0.map(FundingLevel::getName), doc);

        // fundinglevel0_description xpath="//*[local-name()='entity']/*[local-name()='project']/fundingtree//funding_level_0/description"
        addField("fundinglevel0_description", fundingL0.map(FundingLevel::getDescription), doc);

        Optional<FundingLevel> fundingL1 = Optional.ofNullable(p.getFunding()).map(Funding::getLevel1);

        // fundinglevel1_id" xpath="//*[local-name()='entity']/*[local-name()='project']/fundingtree//funding_level_1/id"
        addField("fundinglevel1_id", fundingL1.map(FundingLevel::getId), doc);

        // fundinglevel1_name xpath="//*[local-name()='entity']/*[local-name()='project']/fundingtree//funding_level_1/name"
        addField("fundinglevel1_name", fundingL1.map(FundingLevel::getName), doc);

        // fundinglevel1_description xpath="//*[local-name()='entity']/*[local-name()='project']/fundingtree//funding_level_1/description"
        addField("fundinglevel1_description", fundingL1.map(FundingLevel::getDescription), doc);

        Optional<FundingLevel> fundingL2 = Optional.ofNullable(p.getFunding()).map(Funding::getLevel2);

        // fundinglevel2_id xpath="//*[local-name()='entity']/*[local-name()='project']/fundingtree//funding_level_2/id"
        addField("fundinglevel2_id", fundingL2.map(FundingLevel::getId), doc);

        // fundinglevel2_name xpath="//*[local-name()='entity']/*[local-name()='project']/fundingtree//funding_level_2/name"
        addField("fundinglevel2_name", fundingL2.map(FundingLevel::getName), doc);

        // fundinglevel2_description xpath="//*[local-name()='entity']/*[local-name()='project']/fundingtree//funding_level_2/description"
        addField("fundinglevel2_description", fundingL2.map(FundingLevel::getDescription), doc);

        Optional<Funder> funder = Optional.ofNullable(p.getFunding()).map(Funding::getFunder);

        // funderid xpath="//*[local-name()='entity']/*[local-name()='project']/fundingtree/funder/id"
        addField("funderid", funder.map(Funder::getId), doc);

        // fundershortname xpath="//*[local-name()='entity']/*[local-name()='project']/fundingtree/funder/shortname"
        addField("fundershortname", funder.map(Funder::getShortname), doc);

        // funder
        //      xpath="//*[local-name()='entity']/*[local-name()='project']/fundingtree/funder"
        //      value="concat(./id/text(), '||', ./name/text(), '||', ./shortname/text())"
        addField("funder", new StringJoiner(DELIMITER)
                .add(funder.map(Funder::getId).orElse(null))
                .add(funder.map(Funder::getName).orElse(null))
                .add(funder.map(Funder::getShortname).orElse(null))
                .toString(), doc);

        // country xpath="distinct-values(
        //      *[local-name()='entity']/*/country/@classid |
        //      *[local-name()='entity']/*//rel[./to/@type='organization']/country/@classid |
        //      *[local-name()='entity']//funder/@jurisdiction)"
        addField("country", funder.map(Funder::getJurisdiction).map(Country::getCode), doc);
    }


    private static String mapToOafType(RecordType recordType) {
        switch (recordType) {
            case publication, dataset, other, software -> {
                return "result";
            }
            case datasource -> {
                return "datasource";
            }
            case organization -> {
                return "organization";
            }
            case project -> {
                return "project";
            }
            case person -> {
                return "person";
            }
        }
        throw new IllegalArgumentException("invalid");
    }

    private static void mapResultFields(SolrInputDocument doc, Result r, List<RelatedRecord> links) {

        final List<RelatedRecord> merges = filterLinks(links, "merges");

        // resultdupid xpath="//*[local-name()='entity']/*//children/result/@objidentifier"
        addField("resultdupid",
                merges.stream()
                        .map(RelatedRecord::getHeader)
                        .map(RelatedRecordHeader::getRelatedIdentifier)
                        .toList(), doc);

        // resulttitle xpath="//*[local-name() = 'entity']/*[local-name() ='result']/title | //*[local-name()='entity']/*[local-name()='result']/children/result/title"
        addField("resulttitle", r.getMaintitle(), merges.stream().map(RelatedRecord::getTitle).toList(), doc);

        // resultsubject xpath="distinct-values(//*[local-name()='entity']/*[local-name()='result']/subject)"
        addField("resultsubject", Optional.ofNullable(r.getSubject())
                .map(s -> s.stream()
                        .map(Subject::getValue)
                        .toList())
                .orElse(new ArrayList<>()), doc);

        // resultembargoenddate value="//*[local-name()='entity']/*[local-name()='result']/embargoenddate"
        addField("resultembargoenddate", Optional.ofNullable(r.getEmbargoenddate())
                .map(MappingUtils::normalizeDate), doc);

        // resultembargoendyear value="dnet:extractYear(//*[local-name()='entity']/*[local-name()='result']/embargoenddate)"
        addField("resultembargoendyear", Optional.ofNullable(r.getEmbargoenddate())
                .map(MappingUtils::normalizeDate)
                .map(MappingUtils::extractYear), doc);

        // resulttypeid xpath="//*[local-name()='entity']/*[local-name()='result']/resulttype/@classid"
        addField("resulttypeid", r.getResulttype(), doc);

        // resultlanguagename xpath="//*[local-name()='entity']/*[local-name()='result']/language/@classname"
        addField("resultlanguagename", Optional.ofNullable(r.getLanguage()).map(Language::getLabel), doc);

        // resultpublisher xpath="//*[local-name()='entity']/*[local-name()='result']/*[local-name()='publisher']"
        addField("resultpublisher", r.getPublisher(), doc);

        // resultdescription xpath="//*[local-name()='entity']/*[local-name()='result']//*[local-name()='description']"
        addField("resultdescription", r.getDescription(), doc);

        // resultbestaccessright xpath="distinct-values(//*[local-name()='entity']/*[local-name()='result']/bestaccessright/@classname)"
        addField("resultbestaccessright", Optional.ofNullable(r.getBestaccessright()).map(BestAccessRight::getLabel), doc);

        // resultdateofacceptance value="//*[local-name()='entity']/*[local-name()='result']/dateofacceptance"
        addField("resultdateofacceptance", Optional.ofNullable(r.getPublicationdate())
                .map(MappingUtils::normalizeDate), doc);

        // resultacceptanceyear value="dnet:extractYear(//*[local-name()='entity']/*[local-name()='result']/dateofacceptance)"
        addField("resultacceptanceyear", Optional.ofNullable(r.getPublicationdate())
                .map(MappingUtils::normalizeDate)
                .map(MappingUtils::extractYear), doc);

        // resultauthor xpath="//*[local-name()='entity']/*[local-name()='result']/creator"
        addField("resultauthor", Optional
                .ofNullable(r.getAuthor())
                .map(a -> a.stream()
                        .map(Author::getFullname)
                        .toList())
                .orElse(new ArrayList<>()), doc);

        // authorid xpath="//*[local-name()='entity']/*[local-name()='result']/creator/@*[local-name() != 'rank' and local-name() != 'name' and local-name() != 'surname']"
        addField("authorid", Optional.ofNullable(r.getAuthor())
                        .map(authors -> authors
                                .stream()
                                .map(a -> Optional.ofNullable(a.getId())
                                        .map(id -> Stream.concat(
                                                Stream.of(id),
                                                Optional
                                                    .ofNullable(a.getPid())
                                                    .stream()
                                                    .flatMap(pids -> pids.stream().map(Pid::getValue))))
                                        .orElse(Optional
                                                .ofNullable(a.getPid())
                                                .stream()
                                                .flatMap(pids -> pids.stream().map(Pid::getValue))))
                                .flatMap(p -> p.toList().stream()).toList())
                        .orElse(new ArrayList<>()), doc);

        // orcidtypevalue
        //  xpath="//*[local-name()='entity']/*[local-name()='result']/creator"
        //  value="string-join((./@*[local-name() = 'orcid' or local-name() = 'orcid_pending'], ./@*[local-name() = 'orcid' or local-name() = 'orcid_pending']/local-name()), '||' )"
        addField("orcidtypevalue", Optional.ofNullable(r.getAuthor())
                .stream()
                .flatMap(authors -> authors
                        .stream()
                        .flatMap(a -> Optional
                                .ofNullable(a.getPid())
                                .stream()
                                .flatMap(pids -> pids.stream()
                                        .filter(p -> "orcid".equals(p.getTypeCode()) || "orcid_pending".equals(p.getTypeCode()))
                                        .map(p -> new StringJoiner(DELIMITER)
                                                .add(p.getValue())
                                                .add(p.getTypeCode()).toString()))))
                .toList(), doc);

        // resulthostingdatasource
        //      xpath="//*[local-name()='entity']/*[local-name()='result']/children/instance/*[local-name()='hostedby']"
        //      value="distinct-values(concat(./@id, '||', ./@name))"
        addField("resulthostingdatasource", Optional.ofNullable(r.getInstance())
                .map(instances -> instances
                        .stream()
                        .flatMap(i -> Optional
                                .ofNullable(i.getHostedby())
                                .map(hb -> new StringJoiner(DELIMITER)
                                        .add(hb.getDsId())
                                        .add(hb.getDsName()).toString())
                                .stream())
                        .toList())
                .orElse(new ArrayList<>()), doc);

        // resulthostingdatasourceid xpath="distinct-values(//*[local-name()='entity']/*[local-name()='result']/children/instance/*[local-name()='hostedby']/@id)"
        addField("resulthostingdatasourceid", Optional.ofNullable(r.getInstance())
                .map(instances -> instances
                        .stream()
                        .flatMap(i -> Optional
                                .ofNullable(i.getHostedby())
                                .map(Provenance::getDsId)
                                .stream())
                        .toList())
                .orElse(new ArrayList<>()), doc);

        // collectedfromname xpath="distinct-values(
        //      *[local-name()='entity']/*/*[local-name()='collectedfrom']/@name |
        //      *[local-name()='entity']/*//*[local-name() = 'instance']/*[local-name()='collectedfrom']/@name)"
        addField("collectedfromname", Optional.ofNullable(r.getInstance())
                .map(instances -> instances
                        .stream()
                        .flatMap(i -> Optional
                                .ofNullable(i.getCollectedfrom())
                                .map(Provenance::getDsName)
                                .stream())
                        .toList())
                .orElse(new ArrayList<>()), doc);

        // collectedfromdatasourceid xpath="distinct-values(
        //      *[local-name()='entity']/*/*[local-name()='collectedfrom']/@id |
        //      *[local-name()='entity']/*//*[local-name() = 'instance']/*[local-name()='collectedfrom']/@id)"/>
        addField("collectedfromdatasourceid", Optional.ofNullable(r.getInstance())
                .map(instances -> instances
                        .stream()
                        .flatMap(i ->
                                Optional
                                .ofNullable(i.getCollectedfrom())
                                .map(Provenance::getDsId)
                                .stream())
                        .toList())
                .orElse(new ArrayList<>()), doc);


        //TODO resultdupid result="false" stat="false" tokenizable="false" xpath="//*[local-name()='entity']/*//children/result/@objidentifier"/>

        // instancetypename xpath="distinct-values(//*[local-name()='entity']/*[local-name()='result']/children/instance/*[local-name()='instancetype']/@classname)"
        addField("instancetypename", Optional.ofNullable(r.getInstance())
                .map(instances -> instances
                        .stream()
                        .flatMap(i -> Optional
                                .ofNullable(i.getInstancetype())
                                .stream())
                        .toList())
                .orElse(new ArrayList<>()), doc);

        // externalreflabel xpath="distinct-values(//*[local-name()='entity']/*//children/externalreference/label)"
        addField("externalreflabel", Optional.ofNullable(r.getExternalReference())
                .map(e -> e.stream()
                        .map(ExternalReference::getLabel)
                        .toList())
                .orElse(new ArrayList<>()), doc);

        // resultidentifier xpath="distinct-values(//*[local-name()='entity']/*[local-name()='result']/children/instance/webresource/*[local-name()='url'])"
        addField("resultidentifier", Optional.ofNullable(r.getInstance())
                .map(instances -> instances
                        .stream()
                        .flatMap(i -> Stream.ofNullable(i.getUrl())
                                .flatMap(Collection::stream)
                                .map(s -> StringUtils.left(s, MAX_URL_LENGTH)))
                        .limit(MAX_URLS)
                        .toList())
                .orElse(new ArrayList<>()), doc);

        // resultsource xpath="distinct-values(//*[local-name()='entity']/*[local-name()='result']/source)"
        addField("resultsource", r.getSource(), doc);

        // isgreen" value="//*[local-name()='entity']/*[local-name()='result']/isgreen"
        addField("isgreen", Optional.ofNullable(r.getIsGreen()).map(Objects::toString), doc);

        // openaccesscolor" value="//*[local-name()='entity']/*[local-name()='result']/openaccesscolor"
        addField("openaccesscolor", Optional.ofNullable(r.getOpenAccessColor()).map(Enum::toString), doc);

        // isindiamondjournal value="//*[local-name()='entity']/*[local-name()='result']/isindiamondjournal"
        addField("isindiamondjournal", Optional.ofNullable(r.getIsInDiamondJournal()).map(Objects::toString), doc);

        // publiclyfunded value="//*[local-name()='entity']/*[local-name()='result']/publiclyfunded"
        addField("publiclyfunded", Optional.ofNullable(r.getPubliclyFunded()).map(Objects::toString), doc);

        //peerreviewed value="some $refereed in //*[local-name()='entity']/*[local-name()='result']/children/instance/*[local-name()='refereed']/@classid satisfies ($refereed = '0001')
        addField("peerreviewed", Optional.ofNullable(r.getInstance())
                .map(inst -> inst.stream().anyMatch(i -> "peerReviewed".equals(i.getRefereed())))
                .map(Objects::toString), doc);

        //haslicense value="some $license in //*[local-name()='entity']/*[local-name()='result']/children/instance/*[local-name()='license']/text() satisfies (string-length($license) &gt; 0)
        addField("haslicense", Optional.ofNullable(r.getInstance())
                .map(inst -> inst.stream().anyMatch(i -> StringUtils.isNotEmpty(i.getLicense())))
                .map(Objects::toString), doc);

        //eoscifguidelines xpath="distinct-values(//*[local-name() = 'result']/eoscifguidelines/@code)
        addField("eoscifguidelines", Optional
                .ofNullable(r.getEoscifguidelines())
                .map(e -> e.stream()
                        .map(EoscIfGuidelines::getCode)
                        .toList())
                .orElse(new ArrayList<>()), doc);

        //fos xpath="distinct-values(//*[local-name()='entity']/*[local-name()='result']/subject[@classid='FOS'])"/>
        addField("fos", Optional
                .ofNullable(r.getSubject())
                .map(s -> s.stream()
                        .filter(ss -> "FOS".equals(ss.getTypeCode()))
                        .map(Subject::getValue)
                        .toList())
                .orElse(new ArrayList<>()), doc);

        //foslabel value="concat(./text(), '||', replace(./text(), '^\d+\orgCountries', ''))" xpath="//*[local-name()='entity']/*[local-name()='result']/subject[@classid='FOS']"/>
        addField("foslabel", Optional
                .ofNullable(r.getSubject())
                .map(s -> s.stream()
                        .filter(ss -> "FOS".equals(ss.getTypeCode()))
                        .map(ss -> new StringJoiner(DELIMITER)
                                        .add(ss.getValue())
                                        .add(ss.getValue().replaceFirst("^\\d+\\s", ""))
                                        .toString())
                        .toList())
                .orElse(new ArrayList<>()), doc);

        //sdg xpath="distinct-values(//*[local-name()='entity']/*[local-name()='result']/subject[@classid='SDG'])
        addField("sdg", Optional
                .ofNullable(r.getSubject())
                .map(s -> s.stream()
                        .filter(ss -> "SDG".equals(ss.getTypeCode()))
                        .map(Subject::getValue)
                        .toList())
                .orElse(new ArrayList<>()), doc);

        // country xpath="distinct-values(
        //      *[local-name()='entity']/*/country/@classid |
        //      *[local-name()='entity']/*//rel[./to/@type='organization']/country/@classid |
        //      *[local-name()='entity']//funder/@jurisdiction)"
        final List<String> resultCountries = Optional.ofNullable(r.getCountry())
                .stream()
                .flatMap(countries -> countries.stream().map(Country::getCode))
                .toList();
        final List<String> orgCountries = filterLinks(links, "hasAuthorInstitution")
                .stream()
                .flatMap(rr -> Optional.ofNullable(rr.getCountry()).map(Country::getCode).stream())
                .toList();

        addField("country", Stream.concat(
                Stream.concat(
                        resultCountries.stream(),
                        orgCountries.stream()),
                filterLinks(links, "isProducedBy")
                        .stream()
                        .flatMap(rr -> Optional.ofNullable(rr.getFunding())
                                .flatMap(f -> Optional.ofNullable(f.getFunder())
                                        .flatMap(funder -> Optional.ofNullable(funder.getJurisdiction())
                                                .map(Country::getCode)))
                                .stream())).toList(), doc);

        // countrynojurisdiction xpath="distinct-values(
        //      *[local-name()='entity']/*/country/@classid |
        //      *[local-name()='entity']/*//rel[./to/@type='organization']/country/@classid)"
        addField("countrynojurisdiction", Stream.concat(
                resultCountries.stream(),
                orgCountries.stream()).toList(), doc);

    }
}
