package eu.dnetlib.authentication.security;

import eu.dnetlib.authentication.configuration.AuthenticationConfiguration;
import eu.dnetlib.authentication.services.UserService;
import eu.dnetlib.authentication.utils.EntryPoint;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.config.annotation.method.configuration.EnableMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.config.annotation.web.configurers.AbstractHttpConfigurer;
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.authentication.www.BasicAuthenticationFilter;

@EnableWebSecurity
@EnableMethodSecurity(securedEnabled = true, proxyTargetClass = true)
@Configuration
public class WebSecurityConfig {

    private final EntryPoint entryPoint;
    private final ClientRegistrationRepository clientRegistrationRepository;
    private final UserService userService;
    private final OpenAIREAuthenticationFilter filter;
    private final OpenAIREAuthenticationSuccessHandler successHandler;
    private final OpenAIRELogoutHandler logoutHandler;
    private final OpenAIRELogoutSuccessHandler logoutSuccessHandler;

    @Autowired
    public WebSecurityConfig(UserService userService, EntryPoint entryPoint,
                             ClientRegistrationRepository clientRegistrationRepository,
                             OpenAIREAuthenticationFilter filter,
                             OpenAIREAuthenticationSuccessHandler successHandler,
                             OpenAIRELogoutHandler logoutHandler, OpenAIRELogoutSuccessHandler logoutSuccessHandler) {
        this.userService = userService;
        this.entryPoint = entryPoint;
        this.clientRegistrationRepository = clientRegistrationRepository;
        this.filter = filter;
        this.successHandler = successHandler;
        this.logoutHandler = logoutHandler;
        this.logoutSuccessHandler = logoutSuccessHandler;
    }


    @Bean
    public SecurityFilterChain filterChain(HttpSecurity http) throws Exception {
        http.csrf(AbstractHttpConfigurer::disable);
        http.addFilterBefore(filter, BasicAuthenticationFilter.class);
        http.exceptionHandling(handler -> handler.authenticationEntryPoint(this.entryPoint));
        http.authorizeHttpRequests(auth -> auth.anyRequest().permitAll());
        http.oauth2Login(login -> login
                .clientRegistrationRepository(this.clientRegistrationRepository)
                .authorizationEndpoint(auth -> auth.baseUri(AuthenticationConfiguration.LOGIN_URL))
                .userInfoEndpoint(userInfoEndpointConfig -> userInfoEndpointConfig.oidcUserService(this.userService))
                .successHandler(this.successHandler));
        http.logout(logout -> logout
                .invalidateHttpSession(false)
                .logoutUrl(AuthenticationConfiguration.LOGOUT_URL)
                .addLogoutHandler(this.logoutHandler)
                .logoutSuccessHandler(this.logoutSuccessHandler));
        return http.build();
    }
}
