package eu.dnetlib.authentication.security;

import eu.dnetlib.authentication.configuration.AuthenticationConfiguration;
import eu.dnetlib.authentication.services.UserService;
import eu.dnetlib.uoaauthorizationlibrary.authorization.security.SecurityMode;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.config.annotation.method.configuration.EnableMethodSecurity;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.annotation.web.configuration.EnableWebSecurity;
import org.springframework.security.oauth2.client.registration.ClientRegistrationRepository;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.authentication.www.BasicAuthenticationFilter;

@EnableWebSecurity
@EnableMethodSecurity(securedEnabled = true, proxyTargetClass = true)
@Configuration
public class WebSecurityConfig {

    private final ClientRegistrationRepository clientRegistrationRepository;
    private final UserService userService;
    private final OpenAIREAuthenticationFilter filter;
    private final OpenAIREAuthenticationSuccessHandler successHandler;
    private final OpenAIRELogoutHandler logoutHandler;
    private final OpenAIRELogoutSuccessHandler logoutSuccessHandler;
    private final SecurityMode mode;

    @Autowired
    public WebSecurityConfig(UserService userService,
                             ClientRegistrationRepository clientRegistrationRepository,
                             OpenAIREAuthenticationFilter filter,
                             OpenAIREAuthenticationSuccessHandler successHandler,
                             OpenAIRELogoutHandler logoutHandler,
                             OpenAIRELogoutSuccessHandler logoutSuccessHandler,
                             SecurityMode mode) {
        this.userService = userService;
        this.clientRegistrationRepository = clientRegistrationRepository;
        this.filter = filter;
        this.successHandler = successHandler;
        this.logoutHandler = logoutHandler;
        this.logoutSuccessHandler = logoutSuccessHandler;
        this.mode = mode;
    }


    @Bean
    public SecurityFilterChain filterChain(HttpSecurity http) throws Exception {
        http = this.mode.security(http);
        http.addFilterBefore(filter, BasicAuthenticationFilter.class);
        http.oauth2Login(login -> login
                .clientRegistrationRepository(this.clientRegistrationRepository)
                .authorizationEndpoint(auth -> auth.baseUri(AuthenticationConfiguration.LOGIN_URL))
                .userInfoEndpoint(userInfoEndpointConfig -> userInfoEndpointConfig.oidcUserService(this.userService))
                .successHandler(this.successHandler));
        http.logout(logout -> logout
                .invalidateHttpSession(false)
                .logoutUrl(AuthenticationConfiguration.LOGOUT_URL)
                .addLogoutHandler(this.logoutHandler)
                .logoutSuccessHandler(this.logoutSuccessHandler));
        return http.build();
    }
}
