package eu.dnetlib.enabling.database;

import java.util.Date;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import javax.xml.ws.wsaddressing.W3CEndpointReference;

import eu.dnetlib.enabling.database.rmi.DatabaseException;
import eu.dnetlib.enabling.database.rmi.DatabaseService;
import eu.dnetlib.enabling.resultset.XSLTMappedResultSetFactory;
import eu.dnetlib.enabling.tools.AbstractBaseService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;

public class DatabaseServiceImpl extends AbstractBaseService implements DatabaseService {

	private static final Log log = LogFactory.getLog(DatabaseServiceImpl.class); // NOPMD by marko on 11/24/08 5:02 PM

	private DatabaseServiceCore core;

	private XSLTMappedResultSetFactory xsltResultsetFactory;
	
	private DatabaseBlackBoardNotificationHandler blackBoardNotificationHandler; 

	private ExecutorService threadPool = Executors.newCachedThreadPool();

	@Override
	public W3CEndpointReference dumpTable(final String db, final String table) throws DatabaseException {
		try {
			return core.generateResultSet(db, table, null);
		} catch (Throwable e) {
			throw new DatabaseException(e);
		}
	}

	@Override
	public W3CEndpointReference dumpTableAndLogs(final String db, final String table, final Date from, final Date until) throws DatabaseException {
		try {
			return core.generateResultSet(db, table, from, until);
		} catch (Throwable e) {
			throw new DatabaseException(e);
		}
	}

	@Override
	public void notify(final String subscriptionId, final String topic, final String isId, final String message) {
		log.info("Received notification " + subscriptionId + ", TOPIC: " + topic);
		
		if (topic.contains("BLACKBOARD")) {
			blackBoardNotificationHandler.notified(subscriptionId, topic, isId, message);
			return;
		}
	}

	@Override
	public void importFromEPR(final String db, final W3CEndpointReference epr, final String xslt) throws DatabaseException {

		final W3CEndpointReference mappedEpr = (xslt == null || xslt.isEmpty()) ?
				epr : xsltResultsetFactory.createMappedResultSet(epr, xslt) ;

		try {
			threadPool.execute(new Runnable() {
				@Override
				public void run() {
					try {
						core.importFromResultset(db, mappedEpr);
					} catch (final Exception e) {
						log.error("Error in thread when importing from epr", e);
						throw new RuntimeException(e);
					}
				}
			});
		} catch (RuntimeException e) {
			throw new DatabaseException(e);
		}
	}

	@Required
	public void setCore(final DatabaseServiceCore core) {
		this.core = core;
	}

	@Override
	public W3CEndpointReference searchSQL(final String db, final String sql) throws DatabaseException {
		try {
		return core.generateResultSet(db, sql);
		} catch (Throwable e) {
			throw new DatabaseException(e);
		}
	}
	
	@Override
	public W3CEndpointReference alternativeSearchSQL(String db, String sql, String sqlForSize) throws DatabaseException {
		try {
			return core.generateResultsetWithSize(db, sql, sqlForSize);
		} catch (Throwable e) {
			throw new DatabaseException(e);
		}
	}
	
	@Override
	public void updateSQL(final String db, final String sql) throws DatabaseException {
		core.getDbUtils().executeSql(db, sql);
	}	

	@Override
	public W3CEndpointReference xsltSearchSQL(final String db, final String sql, final String xslt) throws DatabaseException {
		try {
			return xsltResultsetFactory.createMappedResultSet(searchSQL(db, sql), xslt);
		} catch (Throwable e) {
			throw new DatabaseException("Error returning a transformed resultSet", e);
		}
	}

	@Override
	public W3CEndpointReference alternativeXsltSearchSQL(String db, String sql, String sqlForSize, String xslt) throws DatabaseException {
		try {
			return xsltResultsetFactory.createMappedResultSet(alternativeSearchSQL(db, sql, sqlForSize), xslt);
		} catch (Throwable e) {
			throw new DatabaseException("Error returning a transformed resultSet", e);
		}
	}
	
	
	@Override
	public boolean contains(final String db, final String table, final String column, final String value) throws DatabaseException {
		return core.getDbUtils().contains(db, table, column, value);
	}	

	public XSLTMappedResultSetFactory getXsltResultsetFactory() {
		return xsltResultsetFactory;
	}

	@Required
	public void setXsltResultsetFactory(final XSLTMappedResultSetFactory xsltResultsetFactory) {
		this.xsltResultsetFactory = xsltResultsetFactory;
	}

	public ExecutorService getThreadPool() {
		return threadPool;
	}

	public void setThreadPool(ExecutorService threadPool) {
		this.threadPool = threadPool;
	}

	public DatabaseBlackBoardNotificationHandler getBlackBoardNotificationHandler() {
		return blackBoardNotificationHandler;
	}

	@Required
	public void setBlackBoardNotificationHandler(DatabaseBlackBoardNotificationHandler blackBoardNotificationHandler) {
		this.blackBoardNotificationHandler = blackBoardNotificationHandler;
	}

}
