package eu.dnetlib.resultset.inspector;

import static eu.dnetlib.miscutils.collections.MappedCollection.listMap;

import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.ui.ModelMap;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;

import com.google.common.collect.Iterators;
import com.google.common.collect.Ordering;

import eu.dnetlib.enabling.inspector.AbstractInspectorController;
import eu.dnetlib.enabling.resultset.LocalResultSetImpl;
import eu.dnetlib.enabling.resultset.MappedResultSet;
import eu.dnetlib.enabling.resultset.ResultSet;
import eu.dnetlib.enabling.resultset.ResultSetListener;
import eu.dnetlib.enabling.resultset.ResultSetRegistryImpl;
import eu.dnetlib.enabling.resultset.client.IterableResultSetClient;
import eu.dnetlib.enabling.resultset.client.ResultSetClientFactory;
import eu.dnetlib.miscutils.cache.Cache;
import eu.dnetlib.miscutils.cache.CacheElement;
import eu.dnetlib.miscutils.functional.UnaryFunction;
import eu.dnetlib.miscutils.functional.string.EscapeHtml;
import eu.dnetlib.miscutils.functional.xml.ApplyXslt;
import eu.dnetlib.miscutils.functional.xml.IndentXmlString;

@Controller
public class ResultsetInspector extends AbstractInspectorController {

	private static final Log log = LogFactory.getLog(ResultsetInspector.class); // NOPMD by marko on 11/24/08 5:02 PM

	@Resource
	private ResultSetRegistryImpl registry;

	class ResultsetDescription {

		private String id;
		private String type;
		private String handler;
		private boolean open;
		private int size;

		public ResultsetDescription(final String id, final String type, final String handler, final boolean open, final int size) {
			super();
			this.id = id;
			this.type = type;
			this.handler = handler;
			this.open = open;
			this.size = size;
		}

		public String getId() {
			return id;
		}

		public void setId(final String id) {
			this.id = id;
		}

		public String getHandler() {
			return handler;
		}

		public void setHandler(final String handler) {
			this.handler = handler;
		}

		public boolean isOpen() {
			return open;
		}

		public void setOpen(final boolean open) {
			this.open = open;
		}

		public String getType() {
			return type;
		}

		public void setType(final String type) {
			this.type = type;
		}

		public int getSize() {
			return size;
		}

		public void setSize(final int size) {
			this.size = size;
		}
	}

	/**
	 * resultset client.
	 */
	@Resource(name = "resultSetClientFactory")
	private ResultSetClientFactory resultSetClientFactory;

	/**
	 * Default max elements in RS to display
	 */
	private final Integer defaultMax = 100;

	/**
	 * build a resultset and show its content.
	 * 
	 * @param model
	 *            mvc model
	 * @param query
	 *            query (optional)
	 * @param epr
	 *            epr (optional)
	 * @throws Exception
	 */
	@RequestMapping("/inspector/rs.do")
	public void resultset(final ModelMap model,
			@RequestParam(value = "epr", required = false) final String epr,
			@RequestParam(value = "max", required = false) final Integer max) {

		if (epr != null) {

			log.info("building resultset from query or EPR");
			try {
				IterableResultSetClient client = resultSetClientFactory.getClient(epr.trim());
				model.addAttribute("message", "resultset elements listed below:");
				model.addAttribute("iter", Iterators.limit(client.iterator(), max != null ? max : defaultMax));
			} catch (Exception e) {
				model.addAttribute("message", "failed: " + e.getMessage());
				model.addAttribute("iter", "");
			}
		}
		model.addAttribute("epr", epr);
		model.addAttribute("max", max != null ? max : defaultMax);
	}

	/**
	 * Renders a sorded list of local resultset instances taken from the resultset registry eh cache.
	 * 
	 * @param model
	 */
	@RequestMapping(value = "/inspector/resultsets.do")
	public void resultsets(final Model model) {
		final Cache<String, ResultSet> cache = registry.getCache();

		final List<ResultsetDescription> descriptors = listMap(cache.keySet(), new UnaryFunction<ResultsetDescription, String>() {

			@Override
			public ResultsetDescription evaluate(final String arg) {
				final ResultSet rs = cache.get(arg);
				if (rs == null) return new ResultsetDescription("expired", null, null, true, 0);
				String handler = "";
				if (rs instanceof LocalResultSetImpl) {

					final ResultSetListener listener = ((LocalResultSetImpl) rs).getListener();

					handler = listener.getClass().getName();
					if (listener instanceof MappedResultSet) {
						final MappedResultSet mappedResultSet = (MappedResultSet) listener;
						final UnaryFunction<String, String> mapper = mappedResultSet.getMapper();
						handler += "(" + mapper.getClass().getName() + ")";
						if (mapper instanceof ApplyXslt) {
							handler += ": " + ((ApplyXslt) mapper).getXsltName();
						}
					}
				}
				int number = -1;
				try {
					number = rs.getNumberOfResults();
				} catch (Throwable e) {
					// really, who cares
					log.debug("cannot get number of results", e);
				}
				return new ResultsetDescription(arg, rs.getClass().getSimpleName(), handler, rs.isOpen(), number);
			}
		});

		final Ordering<ResultsetDescription> byCreationTime = new Ordering<ResultsetDescription>() {

			@Override
			public int compare(final ResultsetDescription left, final ResultsetDescription right) {
				final CacheElement<ResultSet> rightElement = cache.getElement(right.getId());
				final CacheElement<ResultSet> leftElement = cache.getElement(left.getId());

				if ((leftElement == null) && (rightElement == null)) return 0;
				if ((rightElement == null) && (leftElement != null)) return -1;
				if ((leftElement == null) && (rightElement != null)) return 1;

				return (int) (rightElement.getCreationTime() - leftElement.getCreationTime());
			}
		};

		model.addAttribute("resultsets", byCreationTime.nullsLast().sortedCopy(descriptors));
	}

	@RequestMapping(value = "/inspector/resultset.do")
	public void resultset(final Model model,
			@RequestParam("id") final String id,
			@RequestParam(value = "from", required = false) final Integer from,
			@RequestParam(value = "to", required = false) final Integer to) {

		model.addAttribute("id", id);
		model.addAttribute("from", from);
		model.addAttribute("to", to);

		if ((from != null) && (to != null)) {
			fetchPage(model, id, from, to);
		}
	}

	private void fetchPage(final Model model, final String rsId, final Integer from, final Integer to) {
		final ResultSet rs = registry.getResultSetById(rsId);

		final List<String> res = rs.getResults(from, to);
		model.addAttribute("res", listMap(listMap(res, new IndentXmlString()), new EscapeHtml()));
	}
}
