
package eu.dnetlib.dhp.actionmanager.createunresolvedentities;

import static eu.dnetlib.dhp.actionmanager.Constants.*;
import static eu.dnetlib.dhp.common.SparkSessionSupport.runWithSparkSession;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.spark.SparkConf;
import org.apache.spark.api.java.function.MapFunction;
import org.apache.spark.api.java.function.MapGroupsFunction;
import org.apache.spark.sql.*;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import eu.dnetlib.dhp.actionmanager.createunresolvedentities.model.SDGDataModel;
import eu.dnetlib.dhp.application.ArgumentApplicationParser;
import eu.dnetlib.dhp.schema.common.ModelSupport;
import eu.dnetlib.dhp.schema.oaf.Result;
import eu.dnetlib.dhp.schema.oaf.Subject;
import eu.dnetlib.dhp.schema.oaf.utils.IdentifierFactory;
import eu.dnetlib.dhp.schema.oaf.utils.PidCleaner;
import eu.dnetlib.dhp.schema.oaf.utils.PidType;

public class PrepareSDGSparkJob implements Serializable {

	private static final Logger log = LoggerFactory.getLogger(PrepareSDGSparkJob.class);

	private static final String RESULT_ID_PREFIX = ModelSupport.entityIdPrefix
		.get(Result.class.getSimpleName().toLowerCase()) + IdentifierFactory.ID_PREFIX_SEPARATOR;

	private static final String DOI_PREFIX = "doi_________::";

	public static void main(String[] args) throws Exception {

		String jsonConfiguration = IOUtils
			.toString(
				PrepareSDGSparkJob.class
					.getResourceAsStream(
						"/eu/dnetlib/dhp/actionmanager/createunresolvedentities/prepare_parameters.json"));

		final ArgumentApplicationParser parser = new ArgumentApplicationParser(jsonConfiguration);

		parser.parseArgument(args);

		Boolean isSparkSessionManaged = isSparkSessionManaged(parser);
		log.info("isSparkSessionManaged: {}", isSparkSessionManaged);

		String sourcePath = parser.get("sourcePath");
		log.info("sourcePath: {}", sourcePath);

		final String outputPath = parser.get("outputPath");
		log.info("outputPath: {}", outputPath);

		SparkConf conf = new SparkConf();
		runWithSparkSession(
			conf,
			isSparkSessionManaged,
			spark -> processSDG(spark, sourcePath, outputPath));
	}

	private static void processSDG(SparkSession spark, String sourcePath, String outputPath) {
		readJsonFromPath(spark, sourcePath, SDGDataModel.class)
			.groupByKey((MapFunction<SDGDataModel, String>) PrepareSDGSparkJob::createIdentifier, Encoders.STRING())
			.mapGroups(
				(MapGroupsFunction<String, SDGDataModel, Result>) PrepareSDGSparkJob::getResult,
				Encoders.bean(Result.class))
			.write()
			.mode(SaveMode.Overwrite)
			.option("compression", "gzip")
			.json(outputPath + "/sdg");
	}

	private static String createIdentifier(SDGDataModel v) {
		if (StringUtils.isNotBlank(v.getDoi())) {
			final String doi = PidCleaner.normalizePidValue(PidType.doi.toString(), v.getDoi());
			return RESULT_ID_PREFIX + DOI_PREFIX + IdentifierFactory.md5(doi);
		}
		if (StringUtils.isNotBlank(v.getOaid())) {
			final String oaid = v.getOaid();
			return StringUtils.startsWith(oaid, RESULT_ID_PREFIX) ? oaid : RESULT_ID_PREFIX + oaid;
		}
		throw new RuntimeException("No identifier found for SDGDataModel: " + v);
	}

	private static @NotNull Result getResult(String id, Iterator<SDGDataModel> it) {
		Result r = new Result();
		r.setId(id);
		SDGDataModel first = it.next();
		List<Subject> sbjs = new ArrayList<>();
		sbjs.add(getSubject(first.getSdg(), SDG_CLASS_ID, SDG_CLASS_NAME, UPDATE_SUBJECT_SDG_CLASS_ID));
		it
			.forEachRemaining(
				s -> sbjs
					.add(getSubject(s.getSdg(), SDG_CLASS_ID, SDG_CLASS_NAME, UPDATE_SUBJECT_SDG_CLASS_ID)));
		r.setSubject(sbjs);

		return r;
	}

}
