package eu.dnetlib.broker.openaireAlerts;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import javax.transaction.Transactional;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.stereotype.Service;

import eu.dnetlib.broker.common.elasticsearch.AlertNotificationRepository;
import eu.dnetlib.broker.common.stats.OpenaireDsStat;
import eu.dnetlib.broker.common.stats.OpenaireDsStatRepository;
import eu.dnetlib.broker.common.subscriptions.ConditionOperator;
import eu.dnetlib.broker.common.subscriptions.ConditionParams;
import eu.dnetlib.broker.common.subscriptions.MapCondition;
import eu.dnetlib.broker.common.subscriptions.NotificationFrequency;
import eu.dnetlib.broker.common.subscriptions.NotificationMode;
import eu.dnetlib.broker.common.subscriptions.Subscription;
import eu.dnetlib.broker.common.subscriptions.SubscriptionRepository;
import eu.dnetlib.broker.common.utils.MapValueType;
import eu.dnetlib.broker.events.output.DispatcherManager;

@Service
public class OpenaireAlertsService {

	private static final Log log = LogFactory.getLog(OpenaireAlertsService.class);

	@Autowired
	private JdbcTemplate jdbcTemplate;

	@Autowired
	private OpenaireDsStatRepository dsStatRepository;

	@Autowired
	private SubscriptionRepository subscriptionRepo;

	@Autowired
	private AlertNotificationRepository alertNotificationRepository;

	@Autowired
	private DispatcherManager dispatcher;

	@Transactional
	public OpenaireDsStat updateStats(final OpenaireDsStat stats) {
		dsStatRepository.deleteAlertsByDsId(stats.getDsId());
		return dsStatRepository.save(stats);
	}

	public Optional<Subscription> findSubscription(final String subscrId) {
		return subscriptionRepo.findById(subscrId);
	}

	public String findDatasouceName(final String dsId) {
		return StringUtils.firstNonBlank(jdbcTemplate
				.queryForObject("select name from oa_datasource_stats where id=? and topic ilike 'ALERT/%' limit 1", String.class, dsId), dsId);
	}

	public List<DatasourceWithAlert> findDatasourcesWithAlerts() {
		try {
			final String sql = IOUtils.toString(getClass().getResourceAsStream("/sql/datasourceWithAlerts.sql"), "UTF-8");
			final RowMapper<DatasourceWithAlert> mapper =
					(rs, rowNum) -> new DatasourceWithAlert(rs.getString("id"), rs.getString("name"), rs.getString("rule"), rs.getLong("size"));
			return jdbcTemplate.query(sql, mapper);
		} catch (final Exception e) {
			log.error("Error executing query", e);
			return new ArrayList<>();
		}
	}

	public List<AlertSubscriptionDesc> listAlertSubscriptions(final String email) {
		final Iterable<Subscription> iter = subscriptionRepo.findBySubscriber(email);

		return StreamSupport.stream(iter.spliterator(), false)
				.filter(s -> s.getTopic().startsWith("ALERT/"))
				.map(this::subscriptionDesc)
				.sorted(Comparator.comparing(AlertSubscriptionDesc::getDsName))
				.collect(Collectors.toList());
	}

	private AlertSubscriptionDesc subscriptionDesc(final Subscription s) {
		final String dsId = extractDatasourceId(s);
		final String dsName = findDatasouceName(dsId);

		return new AlertSubscriptionDesc(s.getSubscriptionId(), dsId, dsName, s.getTopic(), s.getCreationDate(), s.getLastNotificationDate(),
				alertNotificationRepository.countBySubscriptionId(s.getSubscriptionId()));
	}

	public Subscription registerSubscription(final String email, final String dsId, final String topic) {
		for (final Subscription s : subscriptionRepo.findBySubscriber(email)) {
			if (topic.equalsIgnoreCase(s.getTopic()) && dsId.equalsIgnoreCase(extractDatasourceId(s))) {
				throw new IllegalArgumentException("Already subscribed");
			}
		}

		final String subscriptionId = "sub-" + UUID.randomUUID();

		final List<MapCondition> conds = new ArrayList<>();
		conds.add(new MapCondition("datasourceId", MapValueType.STRING, ConditionOperator.EXACT, Arrays.asList(new ConditionParams(dsId, null))));

		final Subscription s =
				new Subscription(subscriptionId, email, topic, NotificationFrequency.realtime, NotificationMode.EMAIL, null, new Date(),
						conds);

		return subscriptionRepo.save(s);
	}

	public void sendAlertNotifications(final String dsId) {
		for (final Subscription s : subscriptionRepo.findAll()) {
			if (s.getTopic().startsWith("ALERT/") && s.getConditionsAsList()
					.stream()
					.anyMatch(c -> "datasourcId".equals(c.getField()) && c.getListParams().stream().anyMatch(p -> dsId.equals(p.getValue())))) {

				final long count = alertNotificationRepository.countBySubscriptionId(s.getSubscriptionId());
				if (count > 0) {
					final Map<String, Object> params = new HashMap<>();
					params.put("oa_notifications_total", count);
					params.put("oa_datasource", extractDatasourceId(s));
					dispatcher.sendNotification(s, params);
				}

				s.setLastNotificationDate(new Date());
				subscriptionRepo.save(s);
			}
		}
	}

	private String extractDatasourceId(final Subscription sub) {
		return sub.getConditionsAsList()
				.stream()
				.filter(c -> "datasourceId".equals(c.getField()))
				.map(MapCondition::getListParams)
				.filter(l -> !l.isEmpty())
				.map(l -> l.get(0).getValue())
				.findFirst()
				.orElse("");
	}
}
