package eu.dnetlib.openaire.context;

import java.util.List;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;

@RestController
@CrossOrigin(origins = {
	"*"
})
@ConditionalOnProperty(value = "openaire.exporter.enable.context", havingValue = "true")
@Tag(name = "OpenAIRE Context API", description = "the OpenAIRE Context API")
public class ContextApiController {

	@Autowired
	private ContextApiCore contextApiCore;

	@RequestMapping(value = "/contexts", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "list brief information about all the context profiles", description = "list brief information about all the context profiles")
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public List<ContextSummary> listContexts(@RequestParam(required = false, defaultValue = "") final List<String> type) throws ContextException {
		return contextApiCore.listContexts(type);
	}

	@RequestMapping(value = "/context/{contextId}", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "list the categories defined within a context", description = "list the categories defined within a context")
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public List<CategorySummary> listCategories(
		@PathVariable final String contextId,
		@RequestParam(required = false, defaultValue = "false") final Boolean all) throws ContextException {

		final Boolean allFilter = Optional.ofNullable(all).orElse(false);
		return contextApiCore.listCategories(contextId, allFilter);
	}

	@RequestMapping(value = "/context/category/{categoryId}", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "list the concepts defined within a category", description = "list the concepts defined within a category")
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public List<ConceptSummary> listConcepts(
		@PathVariable final String categoryId,
		@RequestParam(required = false, defaultValue = "false") final Boolean all) throws ContextException {

		final Boolean allFilter = Optional.ofNullable(all).orElse(false);
		return contextApiCore.listConcepts(categoryId, allFilter);
	}

	@RequestMapping(value = "/context/category/concept/{conceptId}", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "list the concepts defined within a category", description = "list the concepts defined within a category")
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public List<ConceptSummary> listSubConcepts(
		@PathVariable final String conceptId,
		@RequestParam(required = false, defaultValue = "false") final Boolean all) throws ContextException {

		final Boolean allFilter = Optional.ofNullable(all).orElse(false);
		return contextApiCore.listSubConcepts(conceptId, allFilter);
	}

}
