package eu.dnetlib.openaire.community;

import static eu.dnetlib.openaire.common.ExporterConstants.C;
import static eu.dnetlib.openaire.common.ExporterConstants.C_CP;
import static eu.dnetlib.openaire.common.ExporterConstants.C_O;
import static eu.dnetlib.openaire.common.ExporterConstants.C_PJ;
import static eu.dnetlib.openaire.common.ExporterConstants.C_SUB;
import static eu.dnetlib.openaire.common.ExporterConstants.C_ZC;
import static eu.dnetlib.openaire.common.ExporterConstants.R;
import static eu.dnetlib.openaire.common.ExporterConstants.W;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.data.domain.Page;
import org.springframework.web.bind.annotation.CrossOrigin;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import eu.dnetlib.openaire.common.AbstractExporterController;
import eu.dnetlib.openaire.exporter.exceptions.CommunityException;
import eu.dnetlib.openaire.exporter.exceptions.ResourceNotFoundException;
import eu.dnetlib.openaire.exporter.model.community.CommunityContentprovider;
import eu.dnetlib.openaire.exporter.model.community.CommunityDetails;
import eu.dnetlib.openaire.exporter.model.community.CommunityOpenAIRECommunities;
import eu.dnetlib.openaire.exporter.model.community.CommunityOrganization;
import eu.dnetlib.openaire.exporter.model.community.CommunityProject;
import eu.dnetlib.openaire.exporter.model.community.CommunitySummary;
import eu.dnetlib.openaire.exporter.model.community.CommunityWritableProperties;
import eu.dnetlib.openaire.exporter.model.community.SubCommunity;
import eu.dnetlib.openaire.exporter.model.community.selectioncriteria.SelectionCriteria;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;

@RestController
@CrossOrigin(origins = {
	"*"
})
@ConditionalOnProperty(value = "openaire.exporter.enable.community", havingValue = "true")
@Tag(name = "OpenAIRE Communities API", description = "the OpenAIRE Community API")
public class CommunityApiController extends AbstractExporterController {

	@Autowired
	private CommunityService communityService;

	@RequestMapping(value = "/community/communities", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "get all community profiles", description = "get all community profiles", tags = {
		C, R
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public List<CommunitySummary> listCommunities() throws CommunityException {
		try {
			return communityService.listCommunities();
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "add a new community profile", description = "add a new community profile", tags = {
		C, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityDetails getCommunity(@RequestBody final CommunityDetails details) throws CommunityException {
		try {
			return communityService.newCommunity(details);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "get community profile", description = "get community profile", tags = {
		C, R
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityDetails getCommunity(@PathVariable final String id) throws CommunityException {
		try {
			return communityService.getCommunity(id);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "update community details", description = "update community details", tags = {
		C, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public void setCommunity(
		@PathVariable final String id,
		@RequestBody final CommunityWritableProperties properties) throws CommunityException {
		try {
			communityService.setCommunity(id, properties);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "delete a community", description = "delete a community", tags = {
		C, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public void deleteCommunity(@PathVariable final String id, @RequestParam(required = false, defaultValue = "false") final boolean recursive)
		throws CommunityException {
		try {
			communityService.deleteCommunity(id, recursive);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/projects/{page}/{size}", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "get community projects", description = "get community projects", tags = {
		C_PJ, R
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public Page<CommunityProject> getCommunityProjects(@PathVariable final String id,
		@PathVariable final Integer page,
		@PathVariable final Integer size,
		@RequestParam(required = false) final String funder,
		@RequestParam(required = false) final String searchFilter,
		@RequestParam(required = false) final String orderBy)
		throws CommunityException {
		try {
			return communityService.getCommunityProjects(id, funder, searchFilter, page, size, orderBy);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/projects", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "associate a project to the community", description = "associate a project to the community", tags = {
		C_PJ, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityProject addCommunityProject(
		@PathVariable final String id,
		@RequestBody final CommunityProject project) throws CommunityException {

		try {
			return communityService.addCommunityProject(id, project);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/projects", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove a project from the community", description = "remove a project from the community", tags = {
		C_PJ, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public void deleteCommunityProject(
		@PathVariable final String id,
		@RequestParam final String projectId) throws CommunityException {
		try {
			communityService.removeCommunityProjects(id, projectId);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/projectList", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "associate a list of project to the community", description = "associate a list of project to the community", tags = {
		C_PJ, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityProject[] addCommunityProjectList(
		@PathVariable final String id,
		@RequestBody final CommunityProject[] projects) throws CommunityException {
		try {
			communityService.addCommunityProjects(id, projects);
			return projects;
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/projectList", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove a list of projects from the community", description = "remove a list of projects from the community", tags = {
		C_PJ, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public void deleteCommunityProjectList(
		@PathVariable final String id,
		@RequestBody final String[] projectIdList) throws CommunityException {
		try {
			communityService.removeCommunityProjects(id, projectIdList);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/funders", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "get the funders of the projects of a community", description = "get the funders of the projects of a community", tags = {
		C_PJ, R
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public List<String> getCommunityFunders(@PathVariable final String id)
		throws CommunityException {
		try {
			return communityService.getCommunityFunders(id);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/contentproviders", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "get the list of content providers associated to a given community", description = "get the list of content providers associated to a given community", tags = {
		C_CP, R
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public List<CommunityContentprovider> getCommunityContentproviders(@PathVariable final String id) throws CommunityException {
		try {
			return communityService.getCommunityContentproviders(id);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/contentproviders", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "associate a content provider to the community", description = "associate a content provider to the community", tags = {
		C_CP, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityContentprovider addCommunityContentprovider(
		@PathVariable final String id,
		@RequestBody final CommunityContentprovider contentprovider) throws CommunityException {

		try {
			communityService.addCommunityContentProviders(id, contentprovider);
			return contentprovider;
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/contentproviders", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove the association between a content provider and the community", description = "remove the association between a content provider and the community", tags = {
		C_CP, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public void removeCommunityContentprovider(
		@PathVariable final String id,
		@RequestParam final String contentproviderId) throws CommunityException {
		try {
			communityService.removeCommunityContentProviders(id, contentproviderId);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/contentprovidersList", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "associate a list of content providers to the community", description = "associate a list of content providers to the community", tags = {
		C_CP, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityContentprovider[] addCommunityContentProvidersList(
		@PathVariable final String id,
		@RequestBody final CommunityContentprovider[] contentprovidersList) throws CommunityException {

		try {
			communityService.addCommunityContentProviders(id, contentprovidersList);
			return contentprovidersList;
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/contentprovidersList", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove a list of content providers from the community", description = "remove a list of content providers from the community", tags = {
		C_CP, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public void deleteCommunityContentProvidersList(
		@PathVariable final String id,
		@RequestBody final String[] contentProviderIdList) throws CommunityException {
		try {
			communityService.removeCommunityContentProviders(id, contentProviderIdList);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	// ADDING CODE FOR COMMUNITY ORGANIZATIONS

	@RequestMapping(value = "/community/{id}/organizations", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "get the list of organizations for a given community", description = "get the list of organizations for a given community", tags = {
		C_O, R
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public List<CommunityOrganization> getCommunityOrganizations(@PathVariable final String id) throws CommunityException {
		try {
			return communityService.getCommunityOrganizations(id);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/organizations", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "associate an organization to the community", description = "associate an organization to the community", tags = {
		C_O, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityOrganization addCommunityOrganization(
		@PathVariable final String id,
		@RequestBody final CommunityOrganization organization) throws CommunityException {
		try {
			communityService.addCommunityOrganizations(id, organization);
			return organization;
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/organizationList", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "associate a list of organizations to the community", description = "associate a list of organizations to the community", tags = {
		C_O, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityOrganization[] addCommunityOrganizationList(
		@PathVariable final String id,
		@RequestBody final CommunityOrganization[] orgs) throws CommunityException {

		try {
			communityService.addCommunityOrganizations(id, orgs);
			return orgs;
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/organizations", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove the association between an organization and the community", description = "remove the association between an organization and the community", tags = {
		C_O, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public void removeCommunityOrganization(
		@PathVariable final String id,
		@RequestParam final String organizationName) throws CommunityException {
		try {
			communityService.removeCommunityOrganizations(id, organizationName);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/organizationList", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove a list of associations between some organizations and the community", description = "remove a list of associations between some organizations and the community", tags = {
		C_O, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public void removeCommunityOrganizationList(
		@PathVariable final String id,
		@RequestBody final String[] orgNames) throws CommunityException {
		try {
			communityService.removeCommunityOrganizations(id, orgNames);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	// **********************

	@RequestMapping(value = "/community/{id}/subjects", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "associate a subject to the community", description = "associate a subject to the community", tags = {
		C, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityDetails addCommunitySubjects(
		@PathVariable final String id,
		@RequestBody final String[] subjects) throws CommunityException {

		try {
			return communityService.addCommunitySubjects(id, subjects);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/subjects", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove subjects from a community", description = "remove subjects from a community", tags = {
		C, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityDetails removeCommunitySubjects(
		@PathVariable final String id,
		@RequestBody final String[] subjects) throws CommunityException {

		try {
			return communityService.removeCommunitySubjects(id, subjects);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/fos", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "associate a fos to the community", description = "associate a fos to the community", tags = {
		C, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityDetails addCommunityFOS(
		@PathVariable final String id,
		@RequestBody final String[] subjects) throws CommunityException {

		try {
			return communityService.addCommunityFOS(id, subjects);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/fos", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove fos from a community", description = "remove fos from a community", tags = {
		C, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityDetails removeCommunityFOS(
		@PathVariable final String id,
		@RequestBody final String[] subjects) throws CommunityException {

		try {
			return communityService.removeCommunityFOS(id, subjects);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/sdg", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "associate a sdg to the community", description = "associate a sdg to the community", tags = {
		C, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityDetails addCommunitySDG(
		@PathVariable final String id,
		@RequestBody final String[] subjects) throws CommunityException {

		try {
			return communityService.addCommunitySDG(id, subjects);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/sdg", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove sdg from a community", description = "remove sdg from a community", tags = {
		C, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityDetails removeCommunitySDG(
		@PathVariable final String id,
		@RequestBody final String[] subjects) throws CommunityException {

		try {
			return communityService.removeCommunitySDG(id, subjects);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/advancedConstraint", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "the set of constraints to be used to extend the association between result and community", description = "the set of constraints to be used to extend the association between result and community", tags = {
		C, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityDetails addAdvancedConstraint(
		@PathVariable final String id,
		@RequestBody final SelectionCriteria advancedConstraint) throws CommunityException {

		try {
			return communityService.addCommunityAdvancedConstraint(id, advancedConstraint);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/advancedConstraint", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove the constraints to extend the association result community from a community", description = "remove the constraints to extend the association result community from a community", tags = {
		C, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityDetails removeAdvancedConstraint(
		@PathVariable final String id) throws CommunityException {

		try {
			return communityService.removeCommunityAdvancedConstraint(id);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/removeConstraint", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "the set of constraints to be used to remove the association between result and community", description = "the set of constraints to be used to remove the association between result and community", tags = {
		C, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityDetails addRemoveConstraint(
		@PathVariable final String id,
		@RequestBody final SelectionCriteria removeConstraint) throws CommunityException {

		try {
			return communityService.addCommunityRemoveConstraint(id, removeConstraint);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/removeConstraint", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove the constraints to remove the association beetween result and community", description = "remove the constraints to remove the association beetween result and community", tags = {
		C, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityDetails removeRemoveConstraint(@PathVariable final String id) throws CommunityException {

		try {
			return communityService.removeCommunityRemoveConstraint(id);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/zenodocommunities", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "associate a Zenodo community to the community", description = "associate a Zenodo community to the community", tags = {
		C_ZC, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityDetails addCommunityZenodoCommunity(
		@PathVariable final String id,
		@RequestParam(required = false, defaultValue = "false") final boolean main,
		@RequestParam final String zenodocommunity) throws CommunityException {

		try {
			return communityService.addCommunityZenodoCommunity(id, zenodocommunity, main);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}

	}

	@RequestMapping(value = "/community/{id}/zenodocommunities", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove a Zenodo community from a community", description = "remove a Zenodo community from a community", tags = {
		C_ZC, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public void removeCommunityZenodoCommunity(
		@PathVariable final String id,
		@RequestParam(required = false, defaultValue = "false") final boolean main,
		@RequestParam final String zenodocommunity) throws CommunityException {
		try {
			communityService.removeCommunityZenodoCommunity(id, zenodocommunity, main);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{zenodoId}/openairecommunities", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "get the list of OpenAIRE communities associated to a given Zenodo community", description = "get the list of OpenAIRE communities associated to a given Zenodo community", tags = {
		C_ZC, R
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public CommunityOpenAIRECommunities getOpenAireCommunities(@PathVariable final String zenodoId) throws CommunityException {
		try {
			final CommunityOpenAIRECommunities res = new CommunityOpenAIRECommunities();
			res.setZenodoid(zenodoId);
			res.setOpenAirecommunitylist(communityService.getOpenAIRECommunitiesByZenodoId(zenodoId));
			return res;
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	// APIs to manage the propagationOrganizationCommunityMap
	@RequestMapping(value = "/propagationOrganizationCommunityMap", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "Get the propagationOrganizationCommunityMap", description = "propagationOrganizationCommunityMap", tags = {
		C_O, R
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public Map<String, Set<String>> getPropagationOrganizationCommunityMap() throws CommunityException {
		try {
			return communityService.getPropagationOrganizationCommunityMap();
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/propagationOrganizations", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "try { return the propagation organizations of a community", description = "try { return the propagation organizations of a community", tags = {
		C_O, R
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public Set<String> getPropagationOrganizationsForCommunity(@PathVariable final String id) throws CommunityException {
		try {
			return communityService.getPropagationOrganizationsForCommunity(id);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/propagationOrganizations", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "add an organization to the propagationOrganizationCommunityMap", description = "add an organization to the propagationOrganizationCommunityMap", tags = {
		C_O, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public Set<String> addPropagationOrganizationForCommunity(@PathVariable final String id,
		@RequestParam final String organizationId) throws CommunityException {

		try {
			return communityService.addPropagationOrganizationForCommunity(id, organizationId.split(","));
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/propagationOrganizations", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "delete an organization to the propagationOrganizationCommunityMap", description = "delete an organization to the propagationOrganizationCommunityMap", tags = {
		C_O, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public Set<String> removePropagationOrganizationForCommunity(@PathVariable final String id,
		@RequestParam final String organizationId) throws CommunityException {

		try {
			return communityService.removePropagationOrganizationForCommunity(id, organizationId.split(","));
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	// APIs to manage the sub communities

	@RequestMapping(value = "/community/{id}/subcommunities", produces = {
		"application/json"
	}, method = RequestMethod.GET)
	@Operation(summary = "get the list of subcommunities for a given community", description = "get the list of subcommunities for a given community", tags = {
		C_SUB, R
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public List<SubCommunity> getSubCommunities(@PathVariable final String id, @RequestParam(required = false, defaultValue = "false") final boolean all)
		throws CommunityException {
		try {
			return communityService.getSubCommunities(id)
				.stream()
				.filter(sc -> all || sc.isBrowsable())
				.collect(Collectors.toList());
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/subcommunities", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "associate a subcommunity to the community", description = "associate a subcommunity to the community", tags = {
		C_SUB, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public SubCommunity addSubCommunity(
		@PathVariable final String id,
		@RequestBody final SubCommunity subcommunity) throws CommunityException {
		try {
			communityService.addSubCommunities(id, subcommunity);
			return subcommunity;
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/subcommunitiesList", produces = {
		"application/json"
	}, method = RequestMethod.POST)
	@Operation(summary = "associate a list of subcommunities to the community", description = "associate a list of subcommunities to the community", tags = {
		C_SUB, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public SubCommunity[] addSubCommunityList(
		@PathVariable final String id,
		@RequestBody final SubCommunity[] subcommunities) throws CommunityException {

		try {
			communityService.addSubCommunities(id, subcommunities);
			return subcommunities;
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/subcommunities", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove the association between a subcommunity and the community", description = "remove the association between a subcommunity and the community", tags = {
		C_SUB, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public void removeSubCommunity(
		@PathVariable final String id,
		@RequestParam final String subCommunityId) throws CommunityException {
		try {
			communityService.removeSubCommunities(id, subCommunityId);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

	@RequestMapping(value = "/community/{id}/subcommunitiesList", produces = {
		"application/json"
	}, method = RequestMethod.DELETE)
	@Operation(summary = "remove a list of associations between some subcommunities and the community", description = "remove a list of associations between some subcommunities and the community", tags = {
		C_SUB, W
	})
	@ApiResponses(value = {
		@ApiResponse(responseCode = "200", description = "OK"),
		@ApiResponse(responseCode = "404", description = "not found"),
		@ApiResponse(responseCode = "500", description = "unexpected error")
	})
	public void removeSubcommunities(
		@PathVariable final String id,
		@RequestBody final String[] subCommunityIdList) throws CommunityException {
		try {
			communityService.removeSubCommunities(id, subCommunityIdList);
		} catch (final ResourceNotFoundException e) {
			throw e;
		} catch (final Throwable e) {
			throw new CommunityException(e);
		}
	}

}
