package eu.dnetlib.organizations.controller;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import eu.dnetlib.common.controller.AbstractDnetController;
import eu.dnetlib.organizations.model.utils.VocabularyTerm;
import eu.dnetlib.organizations.model.view.UserView;
import eu.dnetlib.organizations.repository.UserRepository;
import eu.dnetlib.organizations.repository.readonly.UserViewRepository;
import eu.dnetlib.organizations.utils.DatabaseUtils;

@RestController
public class UserController extends AbstractDnetController {

	@Autowired
	private UserRepository userRepository;

	@Autowired
	private UserViewRepository userViewRepository;
	@Autowired
	private DatabaseUtils dbUtils;

	@PostMapping(value = "/registration_api/newUser")
	public Map<String, Integer> newUser(final @RequestBody List<String> countries, final Authentication authentication) {

		final String email = UserInfo.getEmail(authentication);

		final Map<String, Integer> res = new HashMap<>();

		if (!UserInfo.isNotAuthorized(authentication) || userRepository.existsById(email)) {
			res.put("status", 2);
		} else {
			dbUtils.newUser(email, countries);
			res.put("status", 1);
		}
		return res;
	}

	@GetMapping("/api/users")
	public Iterable<UserView> users(final Authentication authentication) {
		if (UserInfo.isSuperAdmin(authentication)) {
			return userViewRepository.findAll();
		} else if (UserInfo.isNationalAdmin(authentication)) {

			// IMPORTANT: a national admin can manage ONLY the users where ALL the countries are under his control
			final List<UserView> res = new ArrayList<>();
			final List<String> myCountries = dbUtils.listCountriesForUser(UserInfo.getEmail(authentication))
				.stream()
				.map(VocabularyTerm::getValue)
				.collect(Collectors.toList());

			for (final UserView uw : userViewRepository.findAll()) {
				if (uw.getCountries() != null && uw.getCountries().length > 0 && myCountries.containsAll(Arrays.asList(uw.getCountries()))) {
					res.add(uw);
				}
			}
			return res;
		} else {
			return new ArrayList<>();
		}
	}

	@PostMapping("/api/users")
	public Iterable<UserView> save(@RequestBody final UserView userView, final Authentication authentication) {
		if (UserInfo.getEmail(authentication).equals(userView.getEmail())) { throw new RuntimeException("You can't edit your own user"); }
		dbUtils.saveUser(userView);
		return users(authentication);
	}

	@DeleteMapping("/api/users")
	public Iterable<UserView> delete(final @RequestParam String email, final Authentication authentication) {
		if (UserInfo.getEmail(authentication).equals(email)) { throw new RuntimeException("You can't delete your own user"); }
		dbUtils.deleteUser(email);
		return users(authentication);
	}

}
