package eu.dnetlib.organizations.controller;

import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import eu.dnetlib.common.controller.AbstractDnetController;
import eu.dnetlib.organizations.utils.DatabaseUtils;

@RestController
@RequestMapping("/oa_api")
public class OpenaireInternalApiController extends AbstractDnetController {

	@Autowired
	private DatabaseUtils databaseUtils;

	@Value("${openaire.api.https.proxy}")
	private String httpsProxy;

	private static final Log log = LogFactory.getLog(OpenaireInternalApiController.class);

	private final ImportExecution lastExecution = new ImportExecution();

	@GetMapping("/import/dedupEvents")
	public ImportExecution importDedupEvents(final HttpServletRequest req) {
		if (req.getRemoteAddr().equals(httpsProxy)) {
			log.warn("Call received by blaklisted ip (https proxy): " + req.getRemoteAddr());
			throw new RuntimeException("Call received by blaklisted ip (https proxy): " + req.getRemoteAddr());
		}

		synchronized (lastExecution) {
			if (lastExecution.getStatus() != ImportStatus.RUNNING) {
				lastExecution.startNew("Importing dedup events - request from " + req.getRemoteAddr());
				new Thread(() -> {
					try {
						databaseUtils.importDedupEvents();
						lastExecution.complete();
					} catch (final Throwable e) {
						lastExecution.fail(e);
					}
				}).start();
			} else {
				final long now = System.currentTimeMillis();
				return new ImportExecution(null, now, now, ImportStatus.NOT_LAUNCHED, "An other import is running");
			}

		}
		return lastExecution;
	}

	@GetMapping("/import/dedupEvents/status")
	public final ImportExecution statusDedupEvents(final HttpServletRequest req) {
		if (req.getRemoteAddr().equals(httpsProxy)) {
			log.warn("Call received by blaklisted ip (https proxy): " + req.getRemoteAddr());
			throw new RuntimeException("Call received by blaklisted ip (https proxy): " + req.getRemoteAddr());
		}
		return lastExecution;
	}

	@GetMapping("/refresh/fulltextIndex")
	public final List<String> updateFulltextIndex(final HttpServletRequest req) {
		if (req.getRemoteAddr().equals(httpsProxy)) {
			log.warn("Call received by blaklisted ip (https proxy): " + req.getRemoteAddr());
			throw new RuntimeException("Call received by blaklisted ip (https proxy): " + req.getRemoteAddr());
		}
		new Thread(databaseUtils::updateFulltextIndex).start();

		return Arrays.asList("Updating ...");
	}

	class ImportExecution {

		private String id;
		private Long dateStart;
		private Long dateEnd;
		private ImportStatus status = ImportStatus.NOT_YET_STARTED;
		private String message;

		public ImportExecution() {}

		public ImportExecution(final String id, final Long dateStart, final Long dateEnd, final ImportStatus status, final String message) {
			this.id = id;
			this.dateStart = dateStart;
			this.dateEnd = dateEnd;
			this.status = status;
			this.message = message;
		}

		public String getId() {
			return id;
		}

		public void setId(final String id) {
			this.id = id;
		}

		public Long getDateStart() {
			return dateStart;
		}

		public void setDateStart(final Long dateStart) {
			this.dateStart = dateStart;
		}

		public Long getDateEnd() {
			return dateEnd;
		}

		public void setDateEnd(final Long dateEnd) {
			this.dateEnd = dateEnd;
		}

		public ImportStatus getStatus() {
			return status;
		}

		public void setStatus(final ImportStatus status) {
			this.status = status;
		}

		public String getMessage() {
			return message;
		}

		public void setMessage(final String message) {
			this.message = message;
		}

		public void startNew(final String message) {
			setId("import-" + UUID.randomUUID());
			setDateStart(System.currentTimeMillis());
			setDateEnd(null);
			setStatus(ImportStatus.RUNNING);
			setMessage(message);
			log.info(message);
		}

		public void complete() {
			setDateEnd(System.currentTimeMillis());
			setStatus(ImportStatus.SUCCESS);

			final long millis = getDateEnd() - getDateStart();
			setMessage(String
				.format("Import of dedup events completed in %d min, %d sec", TimeUnit.MILLISECONDS.toMinutes(millis), TimeUnit.MILLISECONDS.toSeconds(millis) -
					TimeUnit.MINUTES.toSeconds(TimeUnit.MILLISECONDS.toMinutes(millis))));

			log.info(getMessage());

		}

		public void fail(final Throwable e) {
			setDateEnd(new Date().getTime());
			setStatus(ImportStatus.FAILED);
			setMessage(e.getMessage());
			log.error("Error importing conflicts and duplicates", e);
		}

	}
}

enum ImportStatus {
	SUCCESS,
	FAILED,
	RUNNING,
	NOT_LAUNCHED,
	NOT_YET_STARTED
}
