package eu.dnetlib.data.collector.plugins.fairsharing;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.PriorityBlockingQueue;

import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.Node;
import org.json.JSONObject;
import org.json.XML;

import eu.dnetlib.data.collector.rmi.CollectorServiceException;

public class FairSharingIterator implements Iterator<String> {

	private static final Log log = LogFactory.getLog(FairSharingIterator.class); // NOPMD by marko on 11/24/08 5:02 PM

	private final Queue<String> queue = new PriorityBlockingQueue<>();

	private final String baseUrl;
	private final String authCode;
	private final int pageSize;
	private final Map<String, Element> savedOrgs;

	private String nextUrl;
	private boolean started;

	public FairSharingIterator(final String baseUrl, final String authCode, final int pageSize) {
		this.baseUrl = baseUrl;
		this.authCode = authCode;
		this.pageSize = pageSize;
		this.started = false;
		this.savedOrgs = new HashMap<>();
	}

	private void verifyStarted() {
		if (!this.started) {
			this.started = true;
			try {
				final String url = baseUrl + "/fairsharing_records/?page%5Bnumber%5D=1&page%5Bsize%5D=" + pageSize;
				this.nextUrl = downloadPage(url);
			} catch (final CollectorServiceException e) {
				throw new RuntimeException(e);
			}
		}
	}

	@Override
	public boolean hasNext() {
		synchronized (queue) {
			verifyStarted();
			return !queue.isEmpty();
		}
	}

	@Override
	public String next() {
		synchronized (queue) {
			verifyStarted();
			final String res = queue.poll();
			while (queue.isEmpty() && nextUrl != null && !nextUrl.isEmpty()) {
				try {
					nextUrl = downloadPage(nextUrl);
				} catch (final CollectorServiceException e) {
					throw new RuntimeException(e);
				}
			}
			return res;
		}
	}

	@Override
	public void remove() {}

	private String downloadPage(final String url) throws CollectorServiceException {
		log.debug("Fetching url: " + url);

		final HttpGet req = new HttpGet(url);
		req.addHeader("Accept", "application/json");
		req.addHeader("Content-Type", "application/json");
		req.addHeader("Authorization", "Bearer " + authCode);

		try (final CloseableHttpClient client = HttpClients.createDefault()) {
			try (final CloseableHttpResponse response = client.execute(req)) {
				final String content = IOUtils.toString(response.getEntity().getContent());
				final JSONObject obj = new JSONObject(content);

				for (final Object record : obj.getJSONArray("data")) {
					final String xml = XML.toString(record, "record");
					final Document doc = DocumentHelper.parseText(xml);
					for (final Object o : doc.selectNodes("//grants")) {
						addOrganizationInfo((Element) o);
					}
					fix_null_values(doc);
					queue.add(doc.asXML());
				}

				final JSONObject links = obj.getJSONObject("links");

				return links.isNull("next") ? null : links.getString("next");
			}
		} catch (final Exception e) {
			throw new CollectorServiceException("Error perfoming call for page", e);
		}
	}

	private void addOrganizationInfo(final Element n) throws CollectorServiceException {
		final String orgId = n.valueOf("./organisation_id");

		if (!savedOrgs.containsKey(orgId)) {
			final String url = baseUrl + "/organisations/" + orgId;
			final HttpGet req = new HttpGet(url);
			req.addHeader("Accept", "application/json");
			req.addHeader("Content-Type", "application/json");
			req.addHeader("Authorization", "Bearer " + authCode);

			try (final CloseableHttpClient client = HttpClients.createDefault()) {
				try (final CloseableHttpResponse response = client.execute(req)) {
					final String content = IOUtils.toString(response.getEntity().getContent());
					final JSONObject obj = new JSONObject(content);

					final Document doc = DocumentHelper.parseText(XML.toString(obj.getJSONObject("data"), "extended_info"));
					for (final Object o : doc.selectNodes("//organisation_links")) {
						((Node) o).detach();
					}
					fix_null_values(doc);

					savedOrgs.put(orgId, doc.getRootElement());
				}
			} catch (final Exception e) {
				throw new CollectorServiceException("Error perfoming call for organization", e);
			}
		}

		n.add(savedOrgs.get(orgId).createCopy());
	}

	private void fix_null_values(final Document doc) {
		for (final Object o : doc.selectNodes("//*[text() = 'null']")) {
			((Element) o).setText("");
		}
	}

}
