package eu.dnetlib.data.collector.plugins.gtr2;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.Header;
import org.apache.http.HttpHeaders;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import eu.dnetlib.data.collector.plugins.HttpConnector;
import eu.dnetlib.data.collector.rmi.CollectorServiceRuntimeException;

public class Gtr2Helper {

	private static final Log log = LogFactory.getLog(Gtr2Helper.class); // NOPMD by marko on 11/24/08 5:02 PM

	private static final HttpConnector connector = new HttpConnector();
	private static final DateTimeFormatter simpleDateTimeFormatter = DateTimeFormat.forPattern("yyyy-MM-dd");

	private static final int MAX_ATTEMPTS = 5;

	public static String cleanURL(final String url) {
		String cleaned = url;
		if (cleaned.contains("gtr.gtr")) {
			cleaned = cleaned.replace("gtr.gtr", "gtr");
		}
		if (cleaned.startsWith("http://")) {
			cleaned = cleaned.replaceFirst("http://", "https://");
		}
		return cleaned;
	}

	public static Gtr2ApiResponse loadURL(final String url) {
		final String cleanUrl = cleanURL(url);
		return loadURL(cleanUrl, 0);
	}

	private static Gtr2ApiResponse loadURL(final String cleanUrl, final int attempt) {

		try (final CloseableHttpClient client = HttpClients.createDefault()) {

			final HttpGet req = new HttpGet(cleanUrl);
			req.setHeader(HttpHeaders.ACCEPT, "application/xml");

			try (final CloseableHttpResponse response = client.execute(req)) {

				final Header[] headers = response.getAllHeaders();
				final String content = IOUtils.toString(response.getEntity().getContent());
				final Document doc = DocumentHelper.parseText(content);

				return new Gtr2ApiResponse(doc, headers);
			}
		} catch (final Throwable e) {
			log.error("Error dowloading url: " + cleanUrl + ", attempt = " + attempt, e);
			if (attempt < MAX_ATTEMPTS) {
				try {
					Thread.sleep(30000); // I wait for 30 seconds
				} catch (final InterruptedException e1) {
					throw new CollectorServiceRuntimeException("Error dowloading url: " + cleanUrl, e);
				}
				return loadURL(cleanUrl, attempt + 1);
			} else {
				throw new CollectorServiceRuntimeException("Error dowloading url: " + cleanUrl, e);
			}
		}
	}

	public static DateTime parseDate(final String s) {
		return DateTime.parse(s.contains("T") ? s.substring(0, s.indexOf("T")) : s, simpleDateTimeFormatter);
	}

	public static boolean isAfter(final String d, final DateTime fromDate) {
		return StringUtils.isNotBlank(d) && Gtr2Helper.parseDate(d).isAfter(fromDate);
	}
}
