package eu.dnetlib.data.collector.plugins.croris;

import java.util.Iterator;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.PriorityBlockingQueue;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.dom4j.Document;
import org.dom4j.DocumentHelper;
import org.dom4j.Node;

import eu.dnetlib.data.collector.plugins.utils.JsonUtils;
import eu.dnetlib.data.collector.rmi.CollectorServiceException;

public class CrorisIterator implements Iterator<String> {

	private static final Log log = LogFactory.getLog(CrorisIterator.class);

	private String nextUrl;

	private final int pageSize;

	private final Queue<String> queue = new PriorityBlockingQueue<>();

	public CrorisIterator(final String baseUrl, final int pageSize) {
		this.pageSize = pageSize;
		try {
			this.nextUrl = invokeUrl(baseUrl + "?pageNumber=1&pageSize=" + pageSize);
		} catch (final CollectorServiceException e) {
			throw new IllegalStateException(e);
		}
	}

	@Override
	public boolean hasNext() {
		synchronized (queue) {
			return !queue.isEmpty();
		}
	}

	@Override
	public String next() {
		synchronized (queue) {
			try {
				return queue.poll();
			} finally {
				while (queue.isEmpty() && StringUtils.isNotBlank(nextUrl)) {
					try {
						nextUrl = invokeUrl(nextUrl);
					} catch (final CollectorServiceException e) {
						throw new IllegalStateException(e);
					}
				}
			}
		}
	}

	private String invokeUrl(final String url) throws CollectorServiceException {

		String next = null;

		log.info("Calling url: " + url);

		try (final CloseableHttpClient client = HttpClients.createDefault()) {

			final HttpGet req = new HttpGet(url);

			try (final CloseableHttpResponse response = client.execute(req)) {

				final String content = IOUtils.toString(response.getEntity().getContent());
				final String xml = new JsonUtils().convertToXML(content);

				final Document doc = DocumentHelper.parseText(xml);

				final List<?> list = doc.selectNodes("//projekti");

				for (final Object n : list) {
					queue.add(((Node) n).asXML());
				}

				if (list.size() == pageSize) {
					next = fixUrl(doc.valueOf("//next/href"));
				}
			}

			return next;

		} catch (final Throwable e) {
			log.warn("Error calling url: " + url, e);
			throw new CollectorServiceException("Error calling url: " + url, e);
		}
	}

	private String fixUrl(final String url) {
		return url.replaceAll("\\?", "&").replaceFirst("&", "\\?");
	}

}
