package eu.dnetlib.msro.workflows.hadoop.hbase;

import java.io.IOException;
import java.util.Map;

import com.googlecode.sarasvati.Engine;
import com.googlecode.sarasvati.NodeToken;
import com.googlecode.sarasvati.env.Env;
import eu.dnetlib.data.hadoop.rmi.HadoopBlackboardActions;
import eu.dnetlib.data.hadoop.rmi.HadoopService;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.enabling.resultset.rmi.ResultSetException;
import eu.dnetlib.enabling.tools.blackboard.BlackboardJob;
import eu.dnetlib.miscutils.functional.xml.DnetXsltFunctions;
import eu.dnetlib.msro.rmi.MSROException;
import eu.dnetlib.msro.workflows.nodes.BlackboardJobNode;
import eu.dnetlib.msro.workflows.nodes.ProgressJobNode;
import eu.dnetlib.msro.workflows.nodes.blackboard.BlackboardWorkflowJobListener;
import eu.dnetlib.msro.workflows.resultset.ProcessCountingResultSetFactory;
import eu.dnetlib.msro.workflows.util.ProgressProvider;
import eu.dnetlib.msro.workflows.util.ResultsetProgressProvider;
import eu.dnetlib.msro.workflows.util.WorkflowsConstants;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;

/**
 * Created by claudio on 08/04/16.
 */
public abstract class AbstractHBaseJobNode extends BlackboardJobNode implements ProgressJobNode {

	private static final Log log = LogFactory.getLog(StoreHBaseRecordsJobNode.class); // NOPMD by marko on 11/24/08 5:02 PM

	private final String INPUT_HBASE_TABLE_PARAM = "hbaseTable";
	private final String INPUT_EPR_PARAM = "input_epr";
	private final String INPUT_CLUSTER_PARAM = "cluster";

	private final String XSLT_PARAM = "xslt";

	private final String OUTPUT_HBASE_TABLE_PARAM = "table";
	private final String OUTPUT_CLUSTER_PARAM = "cluster";
	private final String SIMULATION_PARAM = "simulation";

	@Autowired
	protected UniqueServiceLocator serviceLocator;

	protected String inputEprParam;
	protected String hbaseTableProperty;
	protected String cluster;
	protected String mapping;

	protected boolean simulation = false;

	protected ProgressProvider progressProvider;

	protected ProcessCountingResultSetFactory processCountingResultSetFactory;

	protected abstract HadoopBlackboardActions getAction();

	@Override
	protected String obtainServiceId(final NodeToken token) {
		return getServiceLocator().getServiceId(HadoopService.class);
	}

	@Override
	protected void prepareJob(final BlackboardJob job, final NodeToken token) throws Exception {
		log.info("Invoking blackboard method: " + getAction().toString());

		job.setAction(getAction().toString());
		job.getParameters().put(INPUT_EPR_PARAM, DnetXsltFunctions.encodeBase64(prepareEpr(token)));
		job.getParameters().put(XSLT_PARAM, DnetXsltFunctions.encodeBase64(readXslt(getMapping())));
		job.getParameters().put(OUTPUT_HBASE_TABLE_PARAM, tableName(token));
		job.getParameters().put(OUTPUT_CLUSTER_PARAM, cluster(token));
		job.getParameters().put(SIMULATION_PARAM, String.valueOf(isSimulation()));
	}

	@Override
	protected BlackboardWorkflowJobListener generateBlackboardListener(final Engine engine, final NodeToken token) {
		return new BlackboardWorkflowJobListener(engine, token) {

			@Override
			protected void populateEnv(final Env env, final Map<String, String> responseParams) {
				final String count = responseParams.get("count");
				log.info(String.format("%s %s objects to HBase table %s, cluster %s", getAction().toString(), count, tableName(token), cluster(token)));
				env.setAttribute(WorkflowsConstants.MAIN_LOG_PREFIX + getName() + ":count", count);
			}
		};
	}

	protected String tableName(final NodeToken token) {
		if (token.getEnv().hasAttribute(INPUT_HBASE_TABLE_PARAM)) {
			String table = token.getEnv().getAttribute(INPUT_HBASE_TABLE_PARAM);
			log.debug("found override value in wfEnv for 'hbaseTable' param: " + table);
			return table;
		}
		return getPropertyFetcher().getProperty(getHbaseTableProperty());
	}

	protected String cluster(final NodeToken token) {
		if (token.getEnv().hasAttribute(INPUT_CLUSTER_PARAM)) {
			String cluster = token.getEnv().getAttribute("cluster");
			log.debug("found override value in wfEnv for 'cluster' param: " + cluster);
			return cluster;
		}
		return getCluster();
	}

	private String prepareEpr(final NodeToken token) throws ResultSetException {
		final String epr = token.getEnv().getAttribute(inputEprParam);
		final ResultsetProgressProvider resultsetProgressProvider = processCountingResultSetFactory.createProgressProvider(token.getProcess(), epr);

		setProgressProvider(resultsetProgressProvider);

		return resultsetProgressProvider.getEpr().toString();
	}

	private String readXslt(final String profileId) throws IOException, MSROException, ISLookUpException {
		if (StringUtils.isBlank(profileId)) throw new MSROException("missing profile id");

		log.info("loading mapping from profile id: " + profileId);

		final String xquery =
				String.format("/RESOURCE_PROFILE[.//RESOURCE_IDENTIFIER/@value='%s']/BODY/CONFIGURATION/SCRIPT/CODE/*[local-name()='stylesheet']", profileId);
		return serviceLocator.getService(ISLookUpService.class).getResourceProfileByQuery(xquery);
	}

	public String getInputEprParam() {
		return inputEprParam;
	}

	public void setInputEprParam(final String inputEprParam) {
		this.inputEprParam = inputEprParam;
	}

	public String getHbaseTableProperty() {
		return hbaseTableProperty;
	}

	public void setHbaseTableProperty(final String hbaseTableProperty) {
		this.hbaseTableProperty = hbaseTableProperty;
	}

	@Override
	public ProgressProvider getProgressProvider() {
		return progressProvider;
	}

	public void setProgressProvider(final ProgressProvider progressProvider) {
		this.progressProvider = progressProvider;
	}

	public ProcessCountingResultSetFactory getProcessCountingResultSetFactory() {
		return processCountingResultSetFactory;
	}

	@Required
	public void setProcessCountingResultSetFactory(final ProcessCountingResultSetFactory processCountingResultSetFactory) {
		this.processCountingResultSetFactory = processCountingResultSetFactory;
	}

	public String getMapping() {
		return mapping;
	}

	public void setMapping(final String mapping) {
		this.mapping = mapping;
	}

	public String getCluster() {
		return cluster;
	}

	public void setCluster(final String cluster) {
		this.cluster = cluster;
	}

	public boolean isSimulation() {
		return simulation;
	}

	public void setSimulation(final boolean simulation) {
		this.simulation = simulation;
	}

}
