package eu.dnetlib.openaire.directindex.api;

import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.base.Function;
import com.google.common.collect.Lists;
import eu.dnetlib.miscutils.functional.hash.Hashing;
import org.apache.commons.lang.StringUtils;

/**
 * Created by michele on 15/01/16.
 */
public class OpenAIRESubmitterUtils {

	private static final org.apache.commons.logging.Log log = org.apache.commons.logging.LogFactory.getLog(OpenAIRESubmitterUtils.class);

	public Map<String, String> calculateProjectInfo(final String link) {
		final Map<String, String> info = new HashMap<String, String>();
		final String[] arr = link.split("/");
		// info:eu-repo/grantAgreement/EC/FP7/244909/EU/Making Capabilities Work/WorkAble

		if (arr.length > 4) {
			final String acronym = arr.length > 7 ? arr[7] : "";
			final String title = arr.length > 6 ? StringUtils.isNotBlank(arr[6]) ? arr[6] : acronym : "";
			final String jurisdiction = arr.length > 5 ? arr[5] : "";
			final String funderId = calculateFunderId(arr[2], arr[3]);
			info.put("id", calculateProjectId(arr[2], arr[3], arr[4]));
			info.put("funderShortName", arr[2]);
			info.put("fundingName", arr[3]);
			info.put("code", arr[4]);
			info.put("jurisdiction", jurisdiction);
			info.put("title", title);
			info.put("acronym", acronym);
			info.put("funderId", funderId);
			info.put("funderName", calculateFunderName(arr[2]));
			info.put("fundingId", funderId + "::" + arr[3]);
		}
		return info;
	}

	protected String calculateFunderPrefix(final String funderShortName, final String funding){
		switch(funderShortName.toLowerCase()){
		case "conicyt":
			return "conicytf____::";
		case "ec":
			if (funding.equalsIgnoreCase("fp7")) {
				return "corda_______::";
			} else {
				return "corda__h2020::";
			}
		case "hrzz":
		case "mzos":
			return "irb_hr______::";
		case "tara":
			return "taraexp_____::";
		case "tubitak":
			return "tubitakf____::";
		default:
			String prefix = funderShortName.toLowerCase();
			//ensure we have 12 chars
			while(prefix.length() < 12) prefix += "_";
			return prefix +"::";
		}
	}

	protected String calculateProjectId(final String funderShortName, final String funding, final String code) {
		final String suffix = Hashing.md5(code);
		final String funderPrefix = calculateFunderPrefix(funderShortName, funding);
		return funderPrefix + suffix;
	}

	protected String calculateFunderId(final String funderShortName, final String funding) {
		switch (funderShortName.toLowerCase()) {
		case "ec":
			return "ec__________::EC";
		default:
			String prefix = calculateFunderPrefix(funderShortName, funding);
			return prefix + funderShortName.toUpperCase();
		}
	}


	protected String calculateFunderName(final String funderShortName) {

		switch (funderShortName.toLowerCase()) {
		case "aff":
			return "Suomen Akatemia";
		case "arc":
			return "Australian Research Council (ARC)";
		case "conicyt":
			return "Comisión Nacional de Investigación Científica y Tecnológica";
		case "ec":
			return "European Commission";
		case "fct":
			return "Fundação para a Ciência e a Tecnologia, I.P.";
		case "fwf":
			return "Austrian Science Fund (FWF)";
		case "hrzz":
			return "Croatian Science Foundation (CSF)";
		case "mestd":
			return "Ministry of Education, Science and Technological Development of Republic of Serbia";
		case "mzos":
			return "Ministry of Science, Education and Sports of the Republic of Croatia (MSES)";
		case "nhmrc":
			return "National Health and Medical Research Council (NHMRC)";
		case "nih":
			return "National Institutes of Health";
		case "nsf":
			return "National Science Foundation";
		case "nwo":
			return "Netherlands Organisation for Scientific Research (NWO)";
		case "rcuk":
			return "Research Council UK";
		case "sfi":
			return "Science Foundation Ireland";
		case "sgov":
			return "Gobierno de España";
		case "snsf":
			return "Swiss National Science Foundation";
		case "tara":
			return "Tara Expeditions Foundation";
		case "tubitak":
			return "Türkiye Bilimsel ve Teknolojik Araştırma Kurumu";
		case "wt":
			return "Wellcome Trust";
		default:
			log.error("Funder short name '"+funderShortName+"' not managed");
			return "";
		}
	}

	public List<ContextInfo> processContexts(final List<String> list) {
		return Lists.newArrayList(Lists.transform(list, new Function<String, ContextInfo>() {

			@Override
			public ContextInfo apply(final String s) {
				return createContextInfo(s.split("::"), 0);
			}

			private ContextInfo createContextInfo(final String[] arr, final int pos) {
				final StringWriter id = new StringWriter();
				id.write(arr[0]);
				for (int i = 0; i < pos; i++) {
					id.write("::");
					id.write(arr[i + 1]);
				}
				final String elem = (pos == 0) ? "context" : (pos == 1) ? "category" : "concept";
				final ContextInfo info = new ContextInfo(elem, id.toString());
				if ((pos + 1) < arr.length) {
					info.getChildren().add(createContextInfo(arr, pos + 1));
				}
				return info;
			}
		}));

	}

	public class ContextInfo {

		private String elem;
		private String id;
		private List<ContextInfo> children = new ArrayList<ContextInfo>();

		public ContextInfo(final String elem,
				final String id) {
			this.elem = elem;
			this.id = id;
		}

		public String getElem() {
			return elem;
		}

		public void setElem(final String elem) {
			this.elem = elem;
		}

		public String getId() {
			return id;
		}

		public void setId(final String id) {
			this.id = id;
		}

		public List<ContextInfo> getChildren() {
			return children;
		}

		public void setChildren(final List<ContextInfo> children) {
			this.children = children;
		}

		public boolean isRoot() {
			return elem.equals("context");
		}
	}
}