package eu.dnetlib.openaire.directindex.api;

import java.io.IOException;
import java.util.List;

import javax.annotation.Resource;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.app.VelocityEngine;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.bind.annotation.ResponseStatus;

import com.google.common.collect.Lists;
import com.google.gson.Gson;

import eu.dnetlib.common.rmi.DNetRestDocumentation;
import eu.dnetlib.data.index.CloudIndexClient;
import eu.dnetlib.data.index.CloudIndexClientFactory;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import eu.dnetlib.openaire.directindex.objects.ResultEntry;
import eu.dnetlib.openaire.directindex.utils.OafToIndexRecordFactory;

/**
 * Created by michele on 11/11/15.
 */
@Controller
@DNetRestDocumentation
public class OpenaireResultSubmitter {

	private static final Log log = LogFactory.getLog(OpenaireResultSubmitter.class);

	public class IndexDsInfo {

		private final String indexBaseUrl;
		private final String indexDsId;
		private final String format;
		private final String coll;

		public IndexDsInfo(final String indexBaseUrl, final String indexDsId, final String format, final String coll) {
			this.indexBaseUrl = indexBaseUrl;
			this.indexDsId = indexDsId;
			this.format = format;
			this.coll = coll;
		}

		public String getIndexBaseUrl() {
			return indexBaseUrl;
		}

		public String getIndexDsId() {
			return indexDsId;
		}

		public String getFormat() {
			return format;
		}

		public String getColl() {
			return coll;
		}

	}

	@Value(value = "oaf.schema.location")
	private String oafSchemaLocation;

	@Resource
	private UniqueServiceLocator serviceLocator;

	@Resource
	private OafToIndexRecordFactory oafToIndexRecordFactory;

	@Resource
	private RecentResultsQueue recentResultsQueue;

	@Resource(name = "openaireplusApisVelocityEngine")
	private VelocityEngine velocityEngine;

	@Value(value = "${openaire.api.directindex.findSolrIndexUrl.xquery}")
	private ClassPathResource findSolrIndexUrl;

	@Value(value = "${openaire.api.directindex.findIndexDsInfo.xquery}")
	private ClassPathResource findIndexDsInfo;

	@Deprecated
	@RequestMapping(value = { "/api/publications/feedJson", "/api/results/feedJson" }, method = RequestMethod.POST)
	public @ResponseBody String feedObjectJson(@RequestParam(value = "json", required = true) final String json,
			@RequestParam(value = "commit", required = false, defaultValue = "true") final boolean commit) throws DirecIndexApiException {
		final ResultEntry pub = new Gson().fromJson(json, ResultEntry.class);
		return feedObject(pub, commit);
	}

	@RequestMapping(value = { "/api/results/feedObject" }, method = RequestMethod.POST)
	public @ResponseBody String feedResult(@RequestBody final ResultEntry pub,
			@RequestParam(value = "commit", required = false, defaultValue = "true") final boolean commit)
			throws DirecIndexApiException {

		return feedObject(pub, commit);
	}

	@RequestMapping(value = { "/api/publications/feedObject" }, method = RequestMethod.POST)
	public @ResponseBody String feedObject(@RequestBody final ResultEntry pub,
			@RequestParam(value = "commit", required = false, defaultValue = "true") final boolean commit)
			throws DirecIndexApiException {

		final List<IndexDsInfo> idxList;
		try {
			idxList = calculateCurrentIndexDsInfo();
			if (idxList == null
					|| idxList.isEmpty()) { throw new DirecIndexApiException("Cannot add result: " + pub.getAnyId() + " : No public Search Service found"); }
			if (idxList.size() > 1) {
				log.warn("Found more than 1 public search service");
			}
			final String oafRecord = pub.asOafRecord(velocityEngine, serviceLocator.getService(ISLookUpService.class), oafSchemaLocation);

			for (final IndexDsInfo idx : idxList) {
				CloudIndexClient idxClient = null;
				try {
					idxClient = CloudIndexClientFactory.newIndexClient(idx.getIndexBaseUrl(), idx.getColl(), false);
					idxClient.feed(oafRecord, idx.getIndexDsId(), oafToIndexRecordFactory.newTransformer(idx.getFormat()), commit);
				} catch (final Throwable e) {
					throw new DirecIndexApiException("Error adding publication: " + e.getMessage(), e);
				} finally {
					if (idxClient != null) {
						idxClient.close();
					}
				}
			}
			recentResultsQueue.add(oafRecord);
			return pub.getOpenaireId();
		} catch (final Throwable e) {
			log.error("Error saving record", e);
			log.debug(pub.toString());
			throw new DirecIndexApiException("Error adding publication: " + e.getMessage(), e);
		}
	}

	@RequestMapping(value = "/api/result/{openaireId}", method = RequestMethod.DELETE)
	public @ResponseBody boolean deleteResultWithOpenaireId(
			@PathVariable(value = "openaireId") final String openaireId,
			@RequestParam(value = "commit", required = false, defaultValue = "true") final boolean commit) throws DirecIndexApiException {

		return deleteResult(openaireId, commit);
	}

	@RequestMapping(value = "/api/results", method = RequestMethod.DELETE)
	public @ResponseBody boolean deleteResultWithOriginalId(
			@RequestParam(value = "originalId", required = true) final String originalId,
			@RequestParam(value = "collectedFromId", required = true) final String collectedFromId,
			@RequestParam(value = "commit", required = false, defaultValue = "true") final boolean commit) throws Exception {

		final String openaireId = ResultEntry.calculateOpenaireId(originalId, collectedFromId, serviceLocator.getService(ISLookUpService.class));
		return deleteResult(openaireId, commit);
	}

	@Deprecated
	@RequestMapping(value = { "/api/publications/deleteObject", "/api/results/delete" }, method = RequestMethod.POST)
	public @ResponseBody boolean deleteResultPost(
			@RequestParam(value = "originalId", required = true) final String originalId,
			@RequestParam(value = "collectedFromId", required = true) final String collectedFromId,
			@RequestParam(value = "commit", required = false, defaultValue = "true") final boolean commit) throws Exception {

		final String openaireId = ResultEntry.calculateOpenaireId(originalId, collectedFromId, serviceLocator.getService(ISLookUpService.class));
		return deleteResult(openaireId, commit);
	}

	private boolean deleteResult(final String openaireId, final boolean commit) throws DirecIndexApiException {

		final List<IndexDsInfo> idxList;
		try {
			idxList = calculateCurrentIndexDsInfo();

			if (idxList == null
					|| idxList.isEmpty()) { throw new DirecIndexApiException("Cannot delete result: " + openaireId + " : No public Search Service found"); }
			if (idxList.size() > 1) {
				log.warn("Found more than 1 public search service");
			}

			// final String objId = ResultEntry.calculateOpenaireId(originalId, collectedFromId,
			// serviceLocator.getService(ISLookUpService.class));

			for (final IndexDsInfo idx : idxList) {
				CloudIndexClient idxClient = null;
				try {
					idxClient = CloudIndexClientFactory.newIndexClient(idx.getIndexBaseUrl(), idx.getColl(), false);
					idxClient.remove(openaireId, commit);
					log.info("Deleted result with id: " + openaireId + " from: " + idx.getIndexBaseUrl());
				} catch (final Throwable e) {
					throw new DirecIndexApiException("Error deleting publication: " + e.getMessage(), e);
				} finally {
					if (idxClient != null) {
						idxClient.close();
					}
				}
			}
			recentResultsQueue.remove(openaireId);
			return true;
		} catch (IOException | ISLookUpException e) {
			throw new DirecIndexApiException("Error deleting publication: " + e.getMessage(), e);
		}
	}

	@ExceptionHandler(Exception.class)
	@ResponseStatus(value = HttpStatus.INTERNAL_SERVER_ERROR)
	public @ResponseBody ErrorMessage handleException(final Exception e) {
		log.error("Error in direct index API", e);
		return new ErrorMessage(e);
	}

	private List<IndexDsInfo> calculateCurrentIndexDsInfo() throws IOException, ISLookUpException {
		final List<IndexDsInfo> list = Lists.newArrayList();

		final String queryUrl = IOUtils.toString(findSolrIndexUrl.getInputStream());
		final String queryDs = IOUtils.toString(findIndexDsInfo.getInputStream());

		final ISLookUpService lu = serviceLocator.getService(ISLookUpService.class);
		final String indexBaseUrl = lu.getResourceProfileByQuery(queryUrl);

		final List<String> idxDs = lu.quickSearchProfile(queryDs);
		for (final String idx : idxDs) {
			final String[] arr = idx.split("@@@");
			list.add(new IndexDsInfo(indexBaseUrl, arr[0].trim(), arr[1].trim(), arr[2].trim()));
		}
		return list;
	}

	public class ErrorMessage {

		private final String message;
		private final String stacktrace;

		public ErrorMessage(final Exception e) {
			this(e.getMessage(), ExceptionUtils.getStackTrace(e));
		}

		public ErrorMessage(final String message, final String stacktrace) {
			this.message = message;
			this.stacktrace = stacktrace;
		}

		public String getMessage() {
			return this.message;
		}

		public String getStacktrace() {
			return this.stacktrace;
		}

	}

}
