package eu.dnetlib.openaire.directindex.api;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import eu.dnetlib.data.index.CloudIndexClient;
import eu.dnetlib.data.index.CloudIndexClientException;
import eu.dnetlib.data.index.CloudIndexClientFactory;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpException;
import eu.dnetlib.enabling.is.lookup.rmi.ISLookUpService;
import eu.dnetlib.enabling.locators.UniqueServiceLocator;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.io.ClassPathResource;

import javax.annotation.Resource;
import java.io.IOException;
import java.util.List;
import java.util.Map;

public class IndexClientMap {

    private static final Log log = LogFactory.getLog(IndexClientMap.class);

    @Value(value = "${openaire.api.directindex.findSolrIndexUrl.xquery}")
    private ClassPathResource findSolrIndexUrl;

    @Value(value = "${openaire.api.directindex.findIndexDsInfo.xquery}")
    private ClassPathResource findIndexDsInfo;

    @Resource
    private UniqueServiceLocator serviceLocator;

    private Map<IndexDsInfo, CloudIndexClient> clients = Maps.newHashMap();

    public Map<IndexDsInfo, CloudIndexClient> getClients() throws DirecIndexApiException {
        try {
            final List<IndexDsInfo> idxList = calculateCurrentIndexDsInfo();

            if (idxList == null || idxList.isEmpty()) {
                throw new DirecIndexApiException("cannot create index: no public Search Service found");
            }
            if (idxList.size() > 1) {
                log.warn("found more than 1 public search service");
            }

            for(IndexDsInfo i : idxList) {
                if (!clients.containsKey(i)) {
                    clients.put(i, CloudIndexClientFactory.newIndexClient(i.getIndexBaseUrl(), i.getColl(), false));
                }
            }
            return clients;

        } catch (IOException | ISLookUpException | CloudIndexClientException e) {
            throw new DirecIndexApiException(e);
        }
    }

    private List<IndexDsInfo> calculateCurrentIndexDsInfo() throws IOException, ISLookUpException {
        final List<IndexDsInfo> list = Lists.newArrayList();

        final String queryUrl = IOUtils.toString(findSolrIndexUrl.getInputStream());
        final String queryDs = IOUtils.toString(findIndexDsInfo.getInputStream());

        final ISLookUpService lu = serviceLocator.getService(ISLookUpService.class);
        final String indexBaseUrl = lu.getResourceProfileByQuery(queryUrl);

        final List<String> idxDs = lu.quickSearchProfile(queryDs);
        for (final String idx : idxDs) {
            final String[] arr = idx.split("@@@");
            list.add(new IndexDsInfo(indexBaseUrl, arr[0].trim(), arr[1].trim(), arr[2].trim()));
        }
        return list;
    }
}
