package eu.dnetlib.openaire.directindex.api;

import com.google.common.util.concurrent.AbstractScheduledService;
import com.google.common.util.concurrent.Service;
import eu.dnetlib.data.index.CloudIndexClient;
import eu.dnetlib.data.index.CloudIndexClientException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.Resource;
import java.util.Map;
import java.util.concurrent.TimeUnit;

public class ResultSubmitterService {

    private static final Log log = LogFactory.getLog(ResultSubmitterService.class);

    @Resource
    private IndexClientMap clientMap;

    /**
     * Autocommit feature activation flag
     */
    @Value(value = "${openaire.api.directindex.autocommit.active}")
    private boolean autocommitactive;

    /**
     * Autocommit frequency (Seconds)
     */
    @Value(value = "${openaire.api.directindex.autocommit.frequency}")
    private long commitfrquency = 60;

    /**
     * Minimum autocommit frequency
     */
    private static long COMMIT_FREQUENCY_MINVALUE = 5;


    private AbstractScheduledService commitService;


    public ResultSubmitterService() {
        commitService = updateCommitSchedule();
    }

    private AbstractScheduledService updateCommitSchedule() {
        log.info("updating commit schedule");

        if (commitService != null) {
            if (!commitService.state().equals(Service.State.TERMINATED)) {
                log.info("waiting for scheduled commit service to complete ");
                commitService.awaitTerminated();
                if (commitService.state().equals(Service.State.FAILED)) {
                    log.info(commitService.failureCause());
                }
                log.info("stopping commit service");
                commitService.stopAsync();
            }
        }

        final AbstractScheduledService service = new AbstractScheduledService() {
            @Override
            protected void runOneIteration() throws CloudIndexClientException, DirecIndexApiException {
                //TODO perform the commit only if there are pending documents, e.g. query /solr/admin/metrics?group=core&prefix=UPDATE.updateHandler to inspect if there are any.
                if (isAutocommitactive()) {
                       for (Map.Entry<IndexDsInfo, CloudIndexClient> e : clientMap.getClients().entrySet()) {
                        final IndexDsInfo i = e.getKey();
                        final CloudIndexClient client = e.getValue();

                        log.info("performing commit on " + i.getColl());
                        client.commit();
                    }
                }
            }

            @Override
            protected Scheduler scheduler() {
                return Scheduler.newFixedRateSchedule(0, getCommitfrquency(), TimeUnit.SECONDS);
            }
        };
        log.info(String.format("starting commit service, active '%s', frequency '%s'", isAutocommitactive(), getCommitfrquency()));
        service.startAsync();
        return service;

    }

    public boolean isAutocommitactive() {
        return autocommitactive;
    }

    public synchronized void setAutocommitactive(boolean autocommitactive) {
        this.autocommitactive = autocommitactive;
    }

    public long getCommitfrquency() {
        return commitfrquency;
    }

    public synchronized void setCommitfrquency(long commitfrquency) {
        if (commitfrquency < COMMIT_FREQUENCY_MINVALUE) {
            throw new RuntimeException("cannot set autocommit frequency: minimum accepted value (inclusive) is " + COMMIT_FREQUENCY_MINVALUE);
        }
        this.commitfrquency = commitfrquency;
        updateCommitSchedule();
    }

}
