package eu.dnetlib.download.plugin;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;

import com.google.common.collect.Iterables;
import com.google.gson.Gson;
import eu.dnetlib.data.download.rmi.AbstractDownloadPlugin;
import eu.dnetlib.data.download.rmi.DownloadItem;
import eu.dnetlib.data.download.rmi.DownloadPlugin;
import eu.dnetlib.data.download.rmi.DownloadPluginException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class ArxivImportFromFile extends AbstractDownloadPlugin implements DownloadPlugin {

    private static final Log log = LogFactory.getLog(ArxivImportFromFile.class); // NOPMD by marko on 11/24/08 5:02 PM

    /**
     * The base path.
     */
    private String basePath;

    @Override
    public DownloadItem retrieveUrl(final DownloadItem item) {
        if (checkOpenAccess(item) == null) return null;
        final String baseURLs = item.getUrl();
        final List<String> urlsList = new Gson().fromJson(baseURLs, List.class);
        for (final String baseURL : urlsList) {

            if (baseURL.isEmpty() == false && baseURL.trim().startsWith("http://") == true) {
                final String name = StringUtils.substringAfter(baseURL, "abs/").trim();
                if (name == null) {
                    item.setUrl(null);
                    return item;
                }

                final String fileURL = createPath(name);
                if (StringUtils.isBlank(fileURL)) {
                    item.setUrl(null);
                    return item;
                }

                final File f = new File(fileURL);
                if (f.exists()) {
                    if (log.isDebugEnabled()) {
                        log.debug("found path associated to " + item.getIdItemMetadata() + " with path : " + fileURL);
                    }
                    item.setUrl("file://" + fileURL);
                } else {
                    if (log.isDebugEnabled()) {
                        log.debug("NOT found path associated to " + item.getIdItemMetadata()+": "+fileURL);
                    }
                    item.setUrl(null);
                }
                item.setOriginalUrl(baseURL);
                return item;

            } else {
                item.setUrl(null);
            }

        }
        return item;
    }

    public String createPath(final String name) {
        final String regExp = "\\d+\\.\\d+";
        if (name.matches(regExp)) {
            final String[] values = name.split("\\.");
            final Path bsPath = Paths.get(basePath);
            final Path filePath = Paths.get(String.format("%s/%s.pdf", values[0], name));

            final String fileURL = bsPath.resolve(filePath).toString();
            return fileURL;
        } else {
            if (name.contains("/")) {
                final String[] values = name.split("/");
                if (values.length != 2) return null;
                if (values[1].length() > 4) {
                    final String middle = values[1].substring(0, 4);
                    final Path bsPath = Paths.get(basePath);
                    final Path filePath = Paths.get(String.format("%s/%s.pdf", middle, name.replace("/", "")));
                    final String fileURL = bsPath.resolve(filePath).toString();
                    return fileURL;
                }
            }
        }
        return null;
    }

    @Override
    public Iterable<DownloadItem> retrieveUrls(final Iterable<DownloadItem> items) {
        return Iterables.transform(items, input -> retrieveUrl(input));
    }

    @Override
    public String getPluginName() {
        return "ArxivImportFromFile";
    }

    /**
     * @return the basePath
     */
    public String getBasePath() {
        return basePath;
    }

    /**
     * @param basePath the basePath to set
     */
    @Override
    public void setBasePath(final String basePath) {
        this.basePath = basePath;
    }

    @Override
    public String extractURL(String baseURL) throws DownloadPluginException {
        return null;
    }
}
