package eu.dnetlib.data.objectstore.modular.gridFS;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.List;
import java.util.regex.Pattern;

import com.mongodb.BasicDBObject;
import com.mongodb.DBObject;
import com.mongodb.gridfs.GridFS;
import com.mongodb.gridfs.GridFSDBFile;
import com.mongodb.gridfs.GridFSInputFile;
import eu.dnetlib.data.objectstore.modular.ObjectStoreRecord;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStore;
import eu.dnetlib.data.objectstore.rmi.MetadataObjectRecord;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreFile;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreFileNotFoundException;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreServiceException;
import eu.dnetlib.enabling.resultset.ResultSetListener;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

// TODO: Auto-generated Javadoc
/**
 * The Class GridFSObjectStore.
 */
@SuppressWarnings("all")
public class GridFSObjectStore implements ObjectStore {

	/** The Constant log. */
	private static final Log log = LogFactory.getLog(GridFSObjectStore.class);
	/**
	 * The collection.
	 */
	private final GridFS collection;
	/** The id. */
	private String id;
	/** The upsert. */
	private boolean upsert;
	/** The base uri. */
	private String baseURI;

	/**
	 * Instantiates a new grid fs object store.
	 *
	 * @param id
	 *            the id
	 * @param collection
	 *            the collection
	 * @param upsert
	 *            the upsert
	 */
	public GridFSObjectStore(final String id, final GridFS collection, final boolean upsert) {
		this.id = id;
		this.setUpsert(upsert);
		this.collection = collection;

	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStore#getId()
	 */
	@Override
	public String getId() {
		return id;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStore#feedMetadataRecord(java.lang.Iterable, boolean)
	 */
	@Override
	public int feedMetadataRecord(final Iterable<MetadataObjectRecord> records, final boolean incremental) {
		long timestamp = System.currentTimeMillis();
		for (MetadataObjectRecord o : records) {
			if (o == null || o.getRecord() == null) {
				log.debug("Null object metadata record");
				continue;
			}

			GridFSInputFile currentFile = collection.createFile(new ByteArrayInputStream(o.getRecord().getBytes()));
			currentFile.setId(o.getId());
			BasicDBObject metadata = new BasicDBObject();
			metadata.put("id", o.getId());
			metadata.put("mime", o.getMime());
			metadata.put("timestamp", timestamp);
			try {
				String URI = baseURI + "?objectStore=" + URLEncoder.encode(id, "UTF-8") + "&objectId=" + URLEncoder.encode(o.getId(), "UTF-8");
				metadata.put("uri", URI);

			} catch (UnsupportedEncodingException e) {
				log.error("Got an exception during the feed ", e);
			}
			currentFile.setMetaData(metadata);
			currentFile.save();
		}
		return this.getSize();
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStore#feed(java.lang.Iterable, boolean)
	 */
	@Override
	public int feed(final Iterable<ObjectStoreRecord> records, final boolean incremental) {
		long timestamp = System.currentTimeMillis();
		int time = 0;
		for (ObjectStoreRecord o : records) {
			if (o == null || o.getInputStream() == null) {
				if (o != null) {
					log.debug("Null object " + o.getFileMetadata().toJSON());
				} else {
					log.debug("Null Object");
				}
				try {
					Thread.sleep(1000);
				} catch (InterruptedException e) {
					log.error(e);
				}
				continue;
			}
			if (alreadyExist(o.getFileMetadata().getObjectID())) {
				try {
					o.getInputStream().close();
				} catch (IOException e) {
					log.error("Error on closing inputStream ", e);
				}
				continue;
			}
			GridFSInputFile currentFile = collection.createFile(o.getInputStream());
			currentFile.setId(o.getFileMetadata().getObjectID());
			currentFile.setFilename(o.getFileMetadata().getObjectID());
			BasicDBObject metadata = new BasicDBObject();
			metadata.put("id", o.getFileMetadata().getObjectID());
			metadata.put("mime", o.getFileMetadata().getMimeType());
			metadata.put("originalObject", o.getFileMetadata().toJSON());
			metadata.put("timestamp", timestamp);
			try {
				String URI = baseURI + "?objectStore=" + URLEncoder.encode(id, "UTF-8") + "&objectId="
						+ URLEncoder.encode(o.getFileMetadata().getObjectID(), "UTF-8");
				metadata.put("uri", URI);

			} catch (UnsupportedEncodingException e) {
				log.error("Got an exception during the feed ", e);
			}
			currentFile.setMetaData(metadata);
			currentFile.save();
		}
		return getSize();
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStore#deliver(java.lang.Double, java.lang.Double)
	 */
	@Override
	public ResultSetListener deliver(final Long from, final Long until) {
		GridFSObjectstoreResultSetListener resulset = new GridFSObjectstoreResultSetListener();
		resulset.setBaseURI(baseURI);
		resulset.setObjectStoreID(id);
		resulset.setCollection(collection);
		resulset.setFromDate(from);
		resulset.setUntilDate(until);
		return resulset;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStore#deliverIds(java.lang.Iterable)
	 */
	@Override
	public ResultSetListener deliverIds(final Iterable<String> ids) {
		GridFSObjectstoreResultSetListener resulset = new GridFSObjectstoreResultSetListener();
		resulset.setBaseURI(baseURI);
		resulset.setObjectStoreID(id);
		resulset.setCollection(collection);
		resulset.setRecords((List<String>) ids);
		return resulset;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStore#getSize()
	 */
	@Override
	public int getSize() {
		return collection.getFileList().count();
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStore#deleteObject(java.lang.String)
	 */
	@Override
	public void deleteObject(final String objectId) {

	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStore#getObject(java.lang.String)
	 */
	@Override
	public String getObject(final String recordId) {
		return null;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStore#getInterpretation()
	 */
	@Override
	public String getInterpretation() {
		return (String) getMDStoreMetadata().get("interpretation");
	}

	/**
	 * Gets the MD store metadata.
	 *
	 * @return the MD store metadata
	 */
	public DBObject getMDStoreMetadata() {
		return collection.getDB().getCollection("metadataObjectStore").findOne(new BasicDBObject("obsId", Pattern.compile(getId())));
	}

	/**
	 * Gets the base uri.
	 *
	 * @return the base uri
	 */
	public String getBaseURI() {
		return baseURI;
	}

	/**
	 * Sets the base uri.
	 *
	 * @param baseURI
	 *            the new base uri
	 */
	public void setBaseURI(final String baseURI) {
		this.baseURI = baseURI;

	}

	/**
	 * Checks if is upsert.
	 *
	 * @return true, if is upsert
	 */
	public boolean isUpsert() {
		return upsert;
	}

	/**
	 * Sets the upsert.
	 *
	 * @param upsert
	 *            the new upsert
	 */
	public void setUpsert(final boolean upsert) {
		this.upsert = upsert;
	}

	/**
	 * Already exist.
	 *
	 * @param objectId
	 *            the object id
	 * @return true, if successful
	 */
	private boolean alreadyExist(final String objectId) {
		BasicDBObject query = new BasicDBObject("_id", objectId);
		List<GridFSDBFile> file = collection.find(query);
		return file != null && file.size() > 0;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStore#deliverObject(java.lang.String)
	 */
	@Override
	public ObjectStoreFile deliverObject(final String objectId) throws ObjectStoreServiceException {
		BasicDBObject query = new BasicDBObject("_id", objectId);
		List<GridFSDBFile> file = collection.find(query);

		checkSingleItem(objectId, file);

		return ObjectStoreFileUtility.build(file.get(0), baseURI, id);

	}

	/**
	 * Check single item.
	 *
	 * @param objectId
	 *            the object id
	 * @param file
	 *            the file
	 * @throws ObjectStoreFileNotFoundException
	 *             the object store file not found exception
	 */
	private void checkSingleItem(final String objectId, final List<GridFSDBFile> file) throws ObjectStoreFileNotFoundException {
		if (file.isEmpty()) { throw new ObjectStoreFileNotFoundException(String.format("Object file not found with id: %s", objectId)); }
		if (file.size() > 1) { throw new IllegalStateException(String.format("More than one objects found with id: %s", objectId)); }
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see
	 * eu.dnetlib.data.objectstore.modular.connector.ObjectStore#feedObjectRecord(eu.dnetlib.data.objectstore.modular.ObjectStoreRecord)
	 */
	@Override
	public String feedObjectRecord(final ObjectStoreRecord record) throws ObjectStoreServiceException {
		if (record == null || record.getFileMetadata() == null) { throw new ObjectStoreServiceException("Empty input Record"); }

		if (existIDStartsWith(record.getFileMetadata().getObjectID())) {
			log.debug("Object already exist ");
			if (record.getInputStream() != null) {
				try {
					record.getInputStream().close();
				} catch (IOException e) {
					log.debug("Exception happen in closing inputstream " + e);
					throw new ObjectStoreServiceException(e);
				}
			}
			ObjectStoreFile obj = deliverObject(record.getFileMetadata().getObjectID());
			return obj.getURI();
		}
		long timestamp = System.currentTimeMillis();
		String URI = "";

		if (record.getInputStream() == null) { throw new ObjectStoreServiceException("missing inputstream on record " + record.getFileMetadata().getObjectID()); }
		GridFSInputFile currentFile = collection.createFile(record.getInputStream());
		currentFile.setId(record.getFileMetadata().getObjectID());
		currentFile.setFilename(record.getFileMetadata().getObjectID());
		BasicDBObject metadata = new BasicDBObject();
		metadata.put("id", record.getFileMetadata().getObjectID());
		metadata.put("mime", record.getFileMetadata().getMimeType());
		metadata.put("originalObject", record.getFileMetadata().toJSON());

		metadata.put("timestamp", timestamp);
		try {
			URI = baseURI + "?objectStore=" + URLEncoder.encode(id, "UTF-8") + "&objectId="
					+ URLEncoder.encode(record.getFileMetadata().getObjectID(), "UTF-8");
			metadata.put("uri", URI);
			currentFile.setMetaData(metadata);
			currentFile.save();

		} catch (Exception e) {
			log.error(e);
			throw new ObjectStoreServiceException(e);
		} finally {
			try {
				record.getInputStream().close();
			} catch (IOException e) {
				log.error(e);
			}
		}
		return URI;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStore#existIDStartsWith(java.lang.String)
	 */
	@Override
	public boolean existIDStartsWith(final String startId) {
		BasicDBObject query = new BasicDBObject("_id", startId);
		List<GridFSDBFile> file = collection.find(query);
		return file.size() > 0;
	}

	@Override
	public boolean dropContent() throws ObjectStoreServiceException {
		this.collection.remove(new BasicDBObject());
		return true;
	}
}
