package eu.dnetlib.data.objectstore.modular.gridFS;

import java.util.List;
import java.util.regex.Pattern;

import com.mongodb.*;
import com.mongodb.gridfs.GridFS;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStore;
import eu.dnetlib.data.objectstore.modular.connector.ObjectStoreDao;
import eu.dnetlib.data.objectstore.rmi.ObjectStoreServiceException;
import eu.dnetlib.miscutils.collections.MappedCollection;
import eu.dnetlib.miscutils.functional.UnaryFunction;
import org.bson.BSONObject;
import org.springframework.beans.factory.annotation.Required;

// TODO: Auto-generated Javadoc
/**
 * The Class GridFSObjectstoreDaoImpl.
 */
@SuppressWarnings("all")
public class GridFSObjectstoreDaoImpl implements ObjectStoreDao {

	/** The Constant INTERPRETATION. */
	public static final String INTERPRETATION = "interpretation";

	/** The Constant OBJECTSTOREMETADATANAME. */
	private final static String OBJECTSTOREMETADATANAME = "metadataObjectStore";

	/** The Constant OBJECTSTOREIDFIELD. */
	private final static String OBJECTSTOREIDFIELD = "obsId";

	/** The db. */
	private DB db;

	/** The upsert. */
	private boolean upsert;

	/** The object store resturi. */
	private String objectStoreRESTURI;

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStoreDao#getObjectStore(java.lang.String)
	 */
	@Override
	public ObjectStore getObjectStore(String obsId) {
		db.setWriteConcern(WriteConcern.SAFE);
		obsId = obsId.substring(0, 36);
		GridFSObjectStore objectStore = new GridFSObjectStore(obsId, new GridFS(db, obsId), this.upsert);
		objectStore.setBaseURI(objectStoreRESTURI);
		checkIndexes(obsId);
		return objectStore;
	}

	/**
	 * Check indexes.
	 *
	 * @param id
	 *            the id
	 */
	private void checkIndexes(final String id) {
		DBCollection coll = db.getCollection(id + ".files");
		List<DBObject> indexInfo = coll.getIndexInfo();
		boolean hasTimestampIndex = false;
		for (DBObject info : indexInfo) {
			BSONObject boj = (BSONObject) info.get("key");
			if (boj.get("metadata.timestamp") != null) {
				hasTimestampIndex = true;
			}
		}
		if (!hasTimestampIndex) {
			coll.createIndex("metadata.timestamp");
		}
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStoreDao#listObjectStores()
	 */
	@Override
	public List<String> listObjectStores() {
		DBCollection metadata = db.getCollection(OBJECTSTOREMETADATANAME);
		return MappedCollection.listMap(metadata.find(), new UnaryFunction<String, DBObject>() {

			@Override
			public String evaluate(final DBObject object) {
				return (String) object.get(OBJECTSTOREIDFIELD);
			}
		});

	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStoreDao#createObjectStore(java.lang.String, java.lang.String,
	 *      java.lang.String)
	 */
	@Override
	public boolean createObjectStore(final String obsId, final String interpretation, final String basePath) throws ObjectStoreServiceException {
		try {
			DBCollection coll = db.getCollection(OBJECTSTOREMETADATANAME);
			final BasicDBObject obj = new BasicDBObject();
			obj.put(OBJECTSTOREIDFIELD, obsId);
			obj.put(INTERPRETATION, interpretation);
			coll.save(obj);
		} catch (Throwable e) {
			throw new ObjectStoreServiceException("Error on create object store with id: " + obsId, e);
		}
		return true;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStoreDao#updateObjectStore(java.lang.String, java.lang.String)
	 */
	@Override
	public boolean updateObjectStore(final String obsId, final String interpretation) {
		DBCollection coll = db.getCollection(OBJECTSTOREMETADATANAME);
		final BasicDBObject obj = new BasicDBObject();
		obj.put(OBJECTSTOREIDFIELD, obsId);
		obj.put(INTERPRETATION, interpretation);
		coll.update(new BasicDBObject(OBJECTSTOREIDFIELD, obsId), obj, true, false);
		coll.save(obj);
		return true;
	}

	/**
	 * {@inheritDoc}
	 * 
	 * @see eu.dnetlib.data.objectstore.modular.connector.ObjectStoreDao#deleteObjectStore(java.lang.String)
	 */
	@Override
	public boolean deleteObjectStore(String obsId) {
		DBCollection coll = db.getCollection(OBJECTSTOREMETADATANAME);
		coll.remove(new BasicDBObject(OBJECTSTOREIDFIELD, obsId));
		obsId = obsId.substring(0, 36);
		GridFS objectStore = new GridFS(db, obsId);
		Pattern any = Pattern.compile(".");
		BasicDBObject query = new BasicDBObject("md5", any);
		objectStore.remove(query);
		return true;
	}

	@Override
	public boolean dropContent(final String obsId) throws ObjectStoreServiceException {
		return getObjectStore(obsId).dropContent();
	}

	/**
	 * Gets the db.
	 *
	 * @return the db
	 */
	public DB getDb() {
		return db;
	}

	/**
	 * Sets the db.
	 *
	 * @param db
	 *            the new db
	 */
	@Required
	public void setDb(final DB db) {
		this.db = db;
	}

	/**
	 * Checks if is upsert.
	 *
	 * @return true, if is upsert
	 */
	public boolean isUpsert() {
		return upsert;
	}

	/**
	 * Sets the upsert.
	 *
	 * @param upsert
	 *            the new upsert
	 */
	@Required
	public void setUpsert(final boolean upsert) {
		this.upsert = upsert;
	}

	/**
	 * Gets the object store resturi.
	 *
	 * @return the object store resturi
	 */
	public String getObjectStoreRESTURI() {
		return objectStoreRESTURI;
	}

	/**
	 * Sets the object store resturi.
	 *
	 * @param objectStoreRESTURI
	 *            the new object store resturi
	 */
	@Required
	public void setObjectStoreRESTURI(final String objectStoreRESTURI) {
		this.objectStoreRESTURI = objectStoreRESTURI;
	}

}
