package eu.dnetlib.data.objectstore.modular.gridFS;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.Lists;
import com.mongodb.BasicDBObject;
import com.mongodb.DBCursor;
import com.mongodb.DBObject;
import com.mongodb.QueryBuilder;
import com.mongodb.gridfs.GridFS;
import com.mongodb.gridfs.GridFSDBFile;

import eu.dnetlib.enabling.resultset.ResultSet;
import eu.dnetlib.enabling.resultset.ResultSetAware;
import eu.dnetlib.enabling.resultset.ResultSetListener;
import eu.dnetlib.miscutils.collections.MappedCollection;

/**
 * The listener interface for receiving gridFSObjectstoreResultSet events. The class that is interested in processing a
 * gridFSObjectstoreResultSet event implements this interface, and the object created with that class is registered with a component using
 * the component's <code>addGridFSObjectstoreResultSetListener<code> method. When
 * the gridFSObjectstoreResultSet event occurs, that object's appropriate
 * method is invoked.
 *
 * @see GridFSObjectstoreResultSetEvent
 */
public class GridFSObjectstoreResultSetListener implements ResultSetListener, ResultSetAware {

	/** The Constant log. */
	private static final Log log = LogFactory.getLog(GridFSObjectstoreResultSetListener.class);

	/** The from date. */
	private Long fromDate;

	/** The until date. */
	private Long untilDate;

	/** The records. */
	private List<String> records;

	/** The object store id. */
	private String objectStoreID;

	/** The collection. */
	private GridFS collection;

	/** The base uri. */
	private String baseURI;

	/** The current size. */
	private int currentSize = -1;

	/** The current cursor. */
	private DBCursor currentCursor;

	/** The cursor position. */
	private long cursorPosition;

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.enabling.resultset.TypedResultSetListener#getResult(int, int)
	 */
	@Override
	public List<String> getResult(final int from, final int to) {
		log.debug(String.format("ObjectStoreId :%s, from: %d, to: %d", objectStoreID, from, to));
		if (records != null) {
			List<String> ids = Lists.newArrayList();
			for (int i = from; i < to; i++) {
				ids.add(records.get(i));
			}
			QueryBuilder qBuilder = QueryBuilder.start("metadata.id").in(ids);
			DBObject q = qBuilder.get();
			List<GridFSDBFile> out = collection.find(q);

			return MappedCollection.listMap(out, ObjectStoreFileUtility.asJSON(baseURI, objectStoreID));
		} else if ((fromDate != null) && (untilDate != null)) {
			if ((currentCursor == null) || (cursorPosition > from)) {
				createCurrentCursor();
			}
			while (cursorPosition < from) {
				currentCursor.next();
				cursorPosition++;
			}
			ArrayList<GridFSDBFile> out = new ArrayList<GridFSDBFile>();

			for (int i = from; i <= to; i++) {
				if (currentCursor.hasNext()) {
					out.add((GridFSDBFile) currentCursor.next());
					cursorPosition++;
				}
			}

			return MappedCollection.listMap(out, ObjectStoreFileUtility.asJSON(baseURI, objectStoreID));
		}

		throw new IllegalArgumentException("Missing parameters on Delivery must provide either from, to, or ObjectStoreIDs");
	}

	/**
	 * Creates the current cursor.
	 */
	private void createCurrentCursor() {
		BasicDBObject query = new BasicDBObject();
		query.put("$gt", fromDate.doubleValue());
		query.put("$lt", untilDate.doubleValue());
		if (currentCursor != null) {
			currentCursor.close();
		}
		currentCursor = collection.getFileList(new BasicDBObject("metadata.timestamp", query)).sort(new BasicDBObject("_id", 1));
		cursorPosition = 1;

	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.enabling.resultset.TypedResultSetListener#getSize()
	 */
	@Override
	public int getSize() {
		if (currentSize == -1) {
			currentSize = calculateSize();
		}
		return currentSize - 1;
	}

	/**
	 * Calculate size.
	 *
	 * @return the int
	 */
	private int calculateSize() {
		if (records != null) {
			QueryBuilder qBuilder = QueryBuilder.start("metadata.id").in(records);
			DBObject q = qBuilder.get();
			List<GridFSDBFile> out = collection.find(q);
			return out.size();
		} else if ((fromDate != null) && (untilDate != null)) {
			BasicDBObject query = new BasicDBObject();
			query.put("$gt", fromDate.doubleValue());
			query.put("$lt", untilDate.doubleValue());
			return collection.getFileList(new BasicDBObject("metadata.timestamp", query)).size();
		}
		return 0;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see eu.dnetlib.enabling.resultset.ResultSetAware#setResultSet(eu.dnetlib.enabling.resultset.ResultSet)
	 */
	@Override
	public void setResultSet(final ResultSet resultSet) {
		resultSet.close();
	}

	/**
	 * Gets the from date.
	 *
	 * @return the from date
	 */
	public Long getFromDate() {
		return fromDate;
	}

	/**
	 * Sets the from date.
	 *
	 * @param fromdate
	 *            the new from date
	 */
	public void setFromDate(final Long fromdate) {
		this.fromDate = fromdate;
	}

	/**
	 * Gets the until date.
	 *
	 * @return the until date
	 */
	public Long getUntilDate() {
		return untilDate;
	}

	/**
	 * Sets the until date.
	 *
	 * @param untilDate
	 *            the new until date
	 */
	public void setUntilDate(final Long untilDate) {
		this.untilDate = untilDate;
	}

	/**
	 * Gets the records.
	 *
	 * @return the records
	 */
	public List<String> getRecords() {
		return records;
	}

	/**
	 * Sets the records.
	 *
	 * @param records
	 *            the new records
	 */
	public void setRecords(final List<String> records) {
		this.records = records;
	}

	/**
	 * Gets the collection.
	 *
	 * @return the collection
	 */
	public GridFS getCollection() {
		return collection;
	}

	/**
	 * Sets the collection.
	 *
	 * @param collection
	 *            the new collection
	 */
	public void setCollection(final GridFS collection) {
		this.collection = collection;
	}

	/**
	 * Gets the base uri.
	 *
	 * @return the base uri
	 */
	public String getBaseURI() {
		return baseURI;
	}

	/**
	 * Sets the base uri.
	 *
	 * @param baseURI
	 *            the new base uri
	 */
	public void setBaseURI(final String baseURI) {
		this.baseURI = baseURI;
	}

	/**
	 * Gets the object store id.
	 *
	 * @return the object store id
	 */
	public String getObjectStoreID() {
		return objectStoreID;
	}

	/**
	 * Sets the object store id.
	 *
	 * @param objectStoreID
	 *            the new object store id
	 */
	public void setObjectStoreID(final String objectStoreID) {
		this.objectStoreID = objectStoreID;
	}
}
