package eu.dnetlib.data.objectstore.modular.gridFS;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.mongodb.DBObject;
import com.mongodb.gridfs.GridFSDBFile;

import eu.dnetlib.data.objectstore.rmi.ObjectStoreFile;
import eu.dnetlib.data.objectstore.rmi.Protocols;
import eu.dnetlib.miscutils.functional.UnaryFunction;

/**
 * The Class ObjectStoreFileBuilder generates an objectStoreFile bean
 *
 */
public class ObjectStoreFileUtility {

	private static final int KB_SIZE = 1024;

	/** The Constant log. */
	private static final Log log = LogFactory.getLog(ObjectStoreFileUtility.class);

	/**
	 * Builds the.
	 *
	 * @param input
	 *            the input File
	 * @param baseURI
	 *            the base URI to associate to the metadata
	 * @param objectStoreID
	 *            The objectStoreID the input file
	 * @return the object store file
	 */

	public static ObjectStoreFile build(final GridFSDBFile input, final String baseURI, final String objectStoreID) {
		DBObject metadata = input.getMetaData();
		String originalFile = (String) metadata.get("originalObject");
		ObjectStoreFile original = ObjectStoreFile.createObject(originalFile);
		ObjectStoreFile newFile = new ObjectStoreFile();
		newFile.setObjectID((String) metadata.get("id"));
		newFile.setAccessProtocol(Protocols.HTTP);
		newFile.setMimeType((String) metadata.get("mime"));
		newFile.setMd5Sum((String) input.get("md5"));
		newFile.setFileSizeKB(input.getLength() / KB_SIZE);
		if (originalFile != null) {
			newFile.setMetadataRelatedID(original.getMetadataRelatedID());
			if (original.getDownloadedURL() == null || original.getDownloadedURL().length() == 0) {
				newFile.setDownloadedURL(original.getURI());
			} else {
				newFile.setDownloadedURL(original.getDownloadedURL());
			}

		}
		try {
			newFile.setURI(baseURI + "?objectStore=" + URLEncoder.encode(objectStoreID, "UTF-8") + "&objectId="
					+ URLEncoder.encode(newFile.getObjectID(), "UTF-8"));
		} catch (UnsupportedEncodingException e) {
			log.error("Error on Build objectStoreFile ", e);
		}
		return newFile;
	}

	public static UnaryFunction<String, GridFSDBFile> asJSON(final String baseURI, final String objectStoreID) {
		return new UnaryFunction<String, GridFSDBFile>() {

			@Override
			public String evaluate(final GridFSDBFile input) {
				return build(input, baseURI, objectStoreID).toJSON();

			}
		};
	}
}
