package eu.dnetlib.data.mapreduce.hbase.broker.enrich;

import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import eu.dnetlib.data.mapreduce.util.DedupUtils;
import eu.dnetlib.data.mapreduce.util.UpdateMerger;
import eu.dnetlib.data.proto.DedupProtos.Dedup;
import eu.dnetlib.data.proto.DedupProtos.Dedup.RelName;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.TypeProtos.Type;
import org.apache.commons.collections.MapUtils;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.util.Bytes;

/**
 * Created by claudio on 08/07/16.
 */
public class EnrichmentMapper extends TableMapper<ImmutableBytesWritable, ImmutableBytesWritable> {

	private ImmutableBytesWritable outValue;

	private ImmutableBytesWritable outKey;

	@Override
	protected void setup(final Context context) {
		outKey = new ImmutableBytesWritable();
		outValue = new ImmutableBytesWritable();
	}

	@Override
	protected void map(final ImmutableBytesWritable key, final Result value, final Context context) throws IOException, InterruptedException {

		final Map<byte[], byte[]> result = value.getFamilyMap(Bytes.toBytes(Type.result.name()));
		if (MapUtils.isEmpty(result)) {
			context.getCounter(Type.result.name(), "empty family map").increment(1);
			return;
		}

		final Oaf body = UpdateMerger.mergeBodyUpdates(context, result);

		if (body == null) {
			context.getCounter(Type.result.name(), "body null").increment(1);
			return;
		}

		final String mergedInCF = DedupUtils.getDedupCF_mergedIn(Type.result);
		final Map<byte[], byte[]> mergedIn = value.getFamilyMap(Bytes.toBytes(mergedInCF));

		final String outKey = getEmitKey(context, key, mergedIn);

		emit(context, outKey, body);
	}

	private String getEmitKey(final Context context, final ImmutableBytesWritable key, final Map<byte[], byte[]> mergedIn) {
		if (MapUtils.isNotEmpty(mergedIn)) {
			context.getCounter(Type.result.name(), RelName.isMergedIn.name()).increment(1);
			return getRootId(mergedIn, context);
		} else {
			return new String(key.copyBytes());
		}
	}

	private void emit(final Context context, final String key, final Oaf body) throws IOException, InterruptedException {
		outKey.set(Bytes.toBytes(key));
		outValue.set(body.toByteArray());

		context.write(outKey, outValue);
	}

	private String getRootId(final Map<byte[], byte[]> mergedIn, Context context) {
		final HashSet<String> ids = Sets.newHashSet(Iterables.transform(mergedIn.keySet(), new Function<byte[], String>() {

			@Override
			public String apply(final byte[] input) {
				return new String(input);
			}
		}));

		//context.getCounter("duplicate group size", String.valueOf(ids.size())).increment(1);

		try {
			return Iterables.getOnlyElement(ids);
		} catch(IllegalArgumentException e) {
			System.err.println(ids);
			throw e;
		}
	}

}
