package eu.dnetlib.data.mapreduce.hbase.broker;

import java.io.IOException;

import eu.dnetlib.data.mapreduce.util.DedupUtils;
import eu.dnetlib.data.mapreduce.util.OafRowKeyDecoder;
import eu.dnetlib.data.proto.FieldTypeProtos.KeyValue;
import eu.dnetlib.data.proto.OafProtos.Oaf;
import eu.dnetlib.data.proto.PersonProtos.Person;
import org.apache.commons.lang.StringUtils;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.mapreduce.TableMapper;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.io.Text;

public class CalculatePersonDistributionStep1Mapper extends TableMapper<Text, Text> {

	@Override
	protected void setup(final Context context) throws IOException, InterruptedException {
		super.setup(context);
	}

	@Override
	protected void map(final ImmutableBytesWritable key, final Result result, final Context context) throws IOException,
			InterruptedException {

		if (DedupUtils.isRoot(key)) { return; }

		final OafRowKeyDecoder kd = OafRowKeyDecoder.decode(key.copyBytes());
		context.getCounter("MAPPER: processed rows", kd.getType().name()).increment(1);

		switch (kd.getType()) {
		case result:
			processResult(result, context);
			break;
		case datasource:
			processDatasource(result, context);
			break;
		}
	}

	private void processResult(final Result result,
			final Context context) throws IOException, InterruptedException {
		final byte[] body = result.getValue(Bytes.toBytes("result"), Bytes.toBytes("body"));
		if (body == null) {
			context.getCounter("MAPPER: errors", "result body is null").increment(1);
			return;
		}

		final Oaf oaf = Oaf.parseFrom(body);
		final eu.dnetlib.data.proto.ResultProtos.Result r = oaf.getEntity().getResult();
		if (r != null) {
			for (Person p : r.getAuthorList()) {
				for (KeyValue kv : oaf.getEntity().getCollectedfromList()) {
					final String dsId = kv.getKey();
					final String pValue = CalculatePersonDistributionUtils.createPersonValue(p);
					if (StringUtils.isNotBlank(dsId) && StringUtils.isNotBlank(pValue)) {
						context.write(new Text(dsId), new Text(pValue));
						context.getCounter("MAPPER: emitted entities", "person").increment(1);
					} else if (StringUtils.isNotBlank(dsId)) {
						context.getCounter("MAPPER: errors", "person without firstname/secondnames").increment(1);
					} else {
						context.getCounter("MAPPER: errors", "collectedFrom is blank").increment(1);
					}
				}
			}
		}
	}

	private void processDatasource(final Result result,
			final Context context) throws IOException, InterruptedException {
		final byte[] body = result.getValue(Bytes.toBytes("datasource"), Bytes.toBytes("body"));
		if (body == null) {
			context.getCounter("MAPPER: errors", "datasource body is null").increment(1);
			return;
		}

		final Oaf oaf = Oaf.parseFrom(body);
		final eu.dnetlib.data.proto.DatasourceProtos.Datasource ds = oaf.getEntity().getDatasource();
		if (ds != null) {
			context.write(new Text(oaf.getEntity().getId()), new Text(CalculatePersonDistributionUtils.createDsTypeValue(ds)));
			context.getCounter("MAPPER: emitted entities", "datasource").increment(1);
		}
	}
}
